/**
 * Copyright (c) 2024 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.simulation.core.sexec.container

import com.google.inject.Inject
import com.yakindu.base.expressions.interpreter.context.DefaultExecutionContextInitializer
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.Direction
import com.yakindu.base.types.Event
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Package
import com.yakindu.base.types.Property
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sexec.TimeEvent
import com.yakindu.sct.model.sexec.extensions.SExecExtensions
import com.yakindu.sct.model.sgraph.ImportDeclaration
import com.yakindu.sct.model.sgraph.Scope
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sruntime.CompositeSlot
import com.yakindu.sct.model.sruntime.ExecutionEvent
import com.yakindu.sct.model.sruntime.ExecutionSlot
import com.yakindu.sct.model.stext.stext.EventDefinition
import com.yakindu.sct.model.stext.stext.ImportScope
import com.yakindu.sct.model.stext.stext.InterfaceScope
import com.yakindu.sct.model.stext.stext.InternalScope
import com.yakindu.sct.model.stext.stext.OperationDefinition
import com.yakindu.sct.model.stext.stext.VariableDefinition
import org.eclipse.xtext.EcoreUtil2

class SexecExecutionContextInitializer extends DefaultExecutionContextInitializer {
	
	public static final String DEFAULT_SCOPE_NAME = "default"
	public static final String INTERNAL_SCOPE_NAME = "internal"
	public static final String TIME_EVENTS_SCOPE_NAME = "time events"
	public static final String IMPORTS_SCOPE_NAME = "imports"

	
	@Inject protected extension SExecExtensions
	
	def dispatch void define(ExecutionFlow flow, CompositeSlot instance) {
		flow.scopes.forEach[instance.slots += transform]
		flow.features.filter(Operation).filter(o|o.name === "triggerWithoutEvent").forEach [ o |
			instance.slots += createExecutionEvent => [
				name = o.fullyQualifiedName.lastSegment
				fqName = o.fullyQualifiedName.toString
				type = getType(ITypeSystem.VOID)
				direction = Direction.get("IN")
				visible = true
			]
		]
	}
	
	def create it : createCompositeSlot importSlot(ExecutionFlow flow) {
		it => [
			name = IMPORTS_SCOPE_NAME
		]
	}
	
	
	def dispatch ExecutionSlot transform(VariableDefinition it) {
		(it as Property)._transform()
	}
	
	def dispatch ExecutionSlot transform(EventDefinition it) {
		(it as Event)._transform()
	}

	def dispatch ExecutionSlot transform(OperationDefinition it) {
		(it as Operation)._transform()
	}

	def dispatch ExecutionSlot transform(TimeEvent event) {
		createExecutionEvent => [
			name = event.fullyQualifiedName.lastSegment
			fqName = event.fullyQualifiedName.toString
			type = getType(ITypeSystem.INTEGER)
			value = type.initialSlotValue
			direction = Direction.IN
		]
	}
	
	def dispatch ExecutionSlot transform(ExecutionFlow instance) {
		createCompositeSlot => [
			instance.define(it)
		]
	}

	def dispatch ExecutionSlot transform(Scope scope) {
		createCompositeSlot => [
			name = TIME_EVENTS_SCOPE_NAME
			scope.declarations.forEach[decl|slots += decl.transform]
		]
	}
		
	def dispatch ExecutionSlot transform(InternalScope scope) {
		val result = createCompositeSlot => [
			name = INTERNAL_SCOPE_NAME
			scope.declarations.forEach[decl|it.slots += decl.transform]
		]
		// Internal Events should not be visible in simulation view
		result.slots.filter(ExecutionEvent).forEach [
			visible = false
		]
		result
	}

	def dispatch ExecutionSlot transform(InterfaceScope scope) {
		val flow = EcoreUtil2.getContainerOfType(scope, ExecutionFlow)
		val namespace = (flow.sourceElement as Statechart).namespace
		createCompositeSlot => [
			if (scope.name !== null) {
				name = scope.name
				val scopeFqn = scope.fullyQualifiedName.toString
				fqName = if(namespace !== null) namespace + "." + scopeFqn else scopeFqn
			} else {
				name = DEFAULT_SCOPE_NAME
			}
			scope.declarations.forEach[decl|slots += decl.transform]
		]
	}
	
	def dispatch ExecutionSlot transform(ImportScope scope) {

		val composite = importSlot(scope.flow)

		// retrieve namespaces from variable names and create corresponding composite slots
		for (Declaration decl : scope.declarations.filter(ImportDeclaration).map[declaration]) {
			val pkg = EcoreUtil2.getContainerOfType(decl, Package)
			if (pkg !== null) {
				val namespace = pkg.name
				val pkgHeader = pkg.fullyQualifiedName.lastSegment
				val declName = decl.name
				val slot = composite.slots.getSlotFor(pkgHeader)
				val declarationSlot = decl.transform
				declarationSlot.fqName = namespace + "." + declName
				declarationSlot.name = declName
				// only add imported variables/events when they have not yet been imported
				if (!slot.slots.exists[fqName == declarationSlot.fqName]) {
					slot.slots += declarationSlot
				}
			} else {
				composite.slots += decl.transform
			}
		}
		composite
	}
			
}