/**
 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	andreas muelder - itemis AG
 * 
 */
package com.yakindu.sct.domain.java.resource.trafo

import com.google.common.collect.Lists
import com.google.inject.Inject
import com.yakindu.base.expressions.ExpressionBuilder
import com.yakindu.base.types.AnnotationType
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Package
import com.yakindu.base.types.Type
import com.yakindu.base.types.TypeBuilder
import com.yakindu.base.types.TypeParameter
import com.yakindu.base.types.TypeSpecifier
import com.yakindu.base.types.TypesFactory
import com.yakindu.base.types.TypesPackage
import com.yakindu.base.types.TypesUtil
import com.yakindu.base.types.inferrer.ITypeSystemInferrer
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.sct.domain.java.resource.JavaFileLocationAdapter
import com.yakindu.sct.domain.java.resource.trafo.types.ProxyBuilder
import com.yakindu.sct.domain.java.resource.trafo.types.TypeLookup
import com.yakindu.sct.domain.java.typesystem.JavaTypeSystem
import java.lang.annotation.ElementType
import java.lang.annotation.Target
import java.util.Map
import org.eclipse.core.runtime.IAdaptable
import org.eclipse.emf.common.util.URI
import org.eclipse.jdt.core.Flags
import org.eclipse.jdt.core.ICompilationUnit
import org.eclipse.jdt.core.IField
import org.eclipse.jdt.core.ILocalVariable
import org.eclipse.jdt.core.IMethod
import org.eclipse.jdt.core.IType
import org.eclipse.jdt.core.ITypeParameter
import org.eclipse.jdt.core.dom.AST
import org.eclipse.jdt.core.dom.ASTParser
import org.eclipse.jdt.core.dom.ASTVisitor
import org.eclipse.jdt.core.dom.CompilationUnit
import org.eclipse.jdt.core.dom.EnumConstantDeclaration
import org.eclipse.jdt.core.dom.EnumDeclaration
import org.eclipse.jdt.core.dom.Expression
import org.eclipse.jdt.core.dom.NumberLiteral
import org.eclipse.jdt.internal.core.BinaryType

class JavaToTypesTransformation {

	@Inject protected TypeLookup tLookup
	TypesFactory factory = TypesFactory.eINSTANCE
	@Inject protected ITypeSystem ts
	@Inject protected ProxyBuilder proxyBuilder
	@Inject protected extension TypesUtil
	@Inject protected extension TypeBuilder
	@Inject protected extension ExpressionBuilder
	@Inject protected extension ITypeSystemInferrer
	
	Map<String, Integer> enumConstValues

	protected URI uri;

	def createComplexType(IType type) {
		factory.createComplexType => [
			_source_type
			name = type.elementName
			it.typeParameters += type.typeParameters.map[typeParameter]
			it.abstract = isITypeAbstract(type)
			if(type instanceof BinaryType && type.elementName.class !== null){
				var baseType = ts.getType(type.elementName.toLowerCase)
				if(baseType !== null)
					superTypes += baseType.toTypeSpecifier
			}
		]
	}

	def createAnnotationType(IType type) {
		val annotationType = factory.createAnnotationType => [
			name = type.elementName
		]
		val target = type.annotations?.findFirst [
			Target.name.equals(elementName) || Target.simpleName.equals(elementName)
		]
		if (target === null)
			return annotationType
		val memberValue = target?.memberValuePairs.head.value
		val targetStrings = Lists.newArrayList()
		if(memberValue instanceof String) {
			targetStrings.add(memberValue)
		}else if(memberValue instanceof String[]){
			targetStrings.addAll(memberValue as String[])
		}
		targetStrings.forEach [ targetString |
			annotationType.targets += switch ((targetString as String).split("\\.").lastOrNull) {
				case ElementType.ANNOTATION_TYPE.name: TypesPackage.Literals.ANNOTATION_TYPE
				case ElementType.CONSTRUCTOR.name: TypesPackage.Literals.OPERATION
				case ElementType.FIELD.name: TypesPackage.Literals.PROPERTY
				case ElementType.LOCAL_VARIABLE.name: TypesPackage.Literals.PROPERTY
				case ElementType.METHOD.name: TypesPackage.Literals.OPERATION
				case ElementType.PACKAGE.name: TypesPackage.Literals.PACKAGE
				case ElementType.PARAMETER.name: TypesPackage.Literals.PARAMETER
				case ElementType.TYPE.name: TypesPackage.Literals.COMPLEX_TYPE
				case ElementType.TYPE_PARAMETER.name: TypesPackage.Literals.TYPE_PARAMETER
				case ElementType.TYPE_USE.name: TypesPackage.Literals.COMPLEX_TYPE
				default: TypesPackage.Literals.COMPLEX_TYPE
			};

		]
		annotationType
	}

	def createEnum(IType type) {
		factory.createEnumerationType => [
			name = type.elementName
			_source_type
			it.typeParameters += type.typeParameters.map[typeParameter]
			enumerator += type.fields.filter[enumConstant].map[createEnumerator]
		]
	}

	def Type transformHollow(IType it) {
		if (isEnum) {
			return createEnum
		}
		//TODO
//		if (isAnnotation) {
//			return createAnnotationType
//		}
		return createComplexType
	}

	/**
	 * After the type is created by transformHollow, this method adds the properties and operations to the type.
	 * This is needed to counteract cyclic dependencies, for example with the following class:
	 * class A {
	 * 		static class B {
	 * 			A a;
	 * 		}
	 * 		B doStuff();
	 * }
	 * Both types need to exist in the TypeScope before the method can be successfully transformed.
	 */
	protected dispatch def addMembers(ComplexType it, IType type) {
		if(type.isClass) addConstructors(type)
		addMethods(type)
		addFields(type)
		addSuperTypes(type)
	}

	protected dispatch def addMembers(AnnotationType it, IType type) {
		addFields(type)
	}

	protected dispatch def addMembers(Type it, Type type) {
		// Nothing to do
	}

	protected def void addSuperTypes(ComplexType ct, IType type) {
		tLookup.superTypeSignatures(type).forEach [ sig |
			val superType = tLookup.fromSignature(type, sig)?.toTypeSpecifier
			superType.typeArguments += tLookup.getTypeArguments(type.javaProject, sig, type).map[toTypeSpecifier]
			ct.superTypes += superType
		]
	}

	protected def dispatch void addFields(ComplexType ct, IType type) {
		type.fields.filter[type.isInterface || Flags.isPublic(flags)].forEach [ field |
			ct.features += field.property
		]
	}

	protected def dispatch void addFields(AnnotationType ct, IType type) {
		type.methods.forEach [ method |
			//ct.properties += method.property //TODO
		]
	}

	protected def void addMethods(ComplexType ct, IType type) {
		type.methods.filter[type.isInterface || (Flags.isPublic(flags) && !it.constructor)].forEach [ method |
			ct.features += method.operation
		]
	}

	protected def void addConstructors(ComplexType ct, IType type) {
		val ctors = type.methods.filter[it.constructor]
		ctors.filter[Flags.isPublic(flags)].forEach [ ctor |
			ct.features += ctor.operation => [setToConstructor(ct)]
		]
		if (ctors.isEmpty) {
			ct.features += factory.createOperation => [setToConstructor(ct)]
		}
	}

	protected def void setToConstructor(Operation op, ComplexType ct) {
		op.name = JavaTypeSystem.CONSTRUCTOR
		op.static = true
		op.typeSpecifier = ct.toTypeSpecifier
		// for constructors of generic types, create type parameter to invoke target type inference
		ct.typeParameters.forEach [ t, i |
			val tp = factory.createTypeParameter => [name = "T" + i]
			op.typeParameters += tp
			op.typeSpecifier.typeArguments += tp.toTypeSpecifier
		]
	}

	def Type createType(Declaration container, IType iType, IAdaptable cu) {
		enumConstValues = newHashMap
		var isGeneric = !iType.typeParameters.nullOrEmpty
		if (isGeneric) {
			tLookup.push
		}
		if(cu instanceof ICompilationUnit)
			cu.parseAst

		val type = iType.transformHollow
		tLookup.registerType(iType.javaProject, iType.fullyQualifiedName, type)

		if (container instanceof Package) {
			container.member += type
		} else if (container instanceof ComplexType) {
			container.features += type
		}

		iType.types.forEach[subType|createType(type, subType, cu)]

		type.addMembers(iType)
		type.addObjectAsSuperType(iType)

		if (isGeneric) {
			tLookup.pop
		}

		type.eAdapters.add(new JavaFileLocationAdapter(iType.fullyQualifiedName, iType.nameRange))
		type
	}
	
	def parseAst(ICompilationUnit cu) {
		val parser = ASTParser.newParser(AST.JLS17);
		parser.setSource(cu);
		parser.setResolveBindings(true);
		val ast = parser.createAST(null) as CompilationUnit

		ast.accept(new ASTVisitor() {
		    override visit(EnumDeclaration node) {
		        for (Object o : node.enumConstants()) {
		            val c = o as EnumConstantDeclaration ;
		            if (!c.arguments().isEmpty()) {
		                val arg = c.arguments().get(0) as Expression;
		                if (arg instanceof NumberLiteral) {
		                    val value = Integer.parseInt((arg as NumberLiteral).getToken());
		                    enumConstValues.put(c.getName().getIdentifier(), value);
		                }
		            }
		        }
		        return false;
		    }
		});
	}

	protected dispatch def addObjectAsSuperType(ComplexType type, IType iType) {
		if (iType.fullyQualifiedName != Object.name && type.superTypes.isEmpty) {
			type.superTypes +=
				factory.createTypeSpecifier => [type = proxyBuilder.build(Object.name, iType.javaProject)]
		}
	}

	protected dispatch def addObjectAsSuperType(Type type, IType iType) {
		// Nothing to do
	}

	def TypeParameter typeParameter(ITypeParameter typeParameter) {
		val typeParam = createTypeParameter(typeParameter)
		tLookup.registerType(typeParameter.javaProject, typeParameter.elementName, typeParam)
		typeParam
	}

	// TODO: Set the correct bounds		
	def createTypeParameter(ITypeParameter typeParameter) {
		factory.createTypeParameter => [
			name = typeParameter.elementName

		]
	}

	def operation(IMethod method) {
		factory.createOperation => [
			name = method.elementName
			var isGeneric = !method.typeParameters.nullOrEmpty
			if (isGeneric) {
				tLookup.push
				it.typeParameters += method.typeParameters.map[typeParameter]
			}
			typeSpecifier = buildArrayOrRegularTypeSpecifier(tLookup.resolveType(method), tLookup.arrayDepth(method))
			typeSpecifier.typeArguments += tLookup.getTypeArguments(method).map[toTypeSpecifier]
			method.parameters.forEach[param|parameters += param.opParameter]

			if (isGeneric) {
				tLookup.pop
			}

			static = Flags.isStatic(method.flags)
			eAdapters.add(new JavaFileLocationAdapter(method.declaringType.fullyQualifiedName, method.nameRange))
		]
	}

	def opParameter(ILocalVariable variable) {
		factory.createParameter => [
			typeSpecifier = buildArrayOrRegularTypeSpecifier(tLookup.resolveType(variable),
				tLookup.arrayDepth(variable))
			typeSpecifier.typeArguments += tLookup.getTypeArguments(variable).map[toTypeSpecifier]
			name = variable.elementName
		]
	}

	def TypeSpecifier buildArrayOrRegularTypeSpecifier(Type innerType, int arrayDepth) {
		var aD = arrayDepth
		var ts = factory.createTypeSpecifier => [
			it.type = innerType
		]
		while (aD > 0) {
			val innerTs = ts
			ts = factory.createArrayTypeSpecifier => [
				it.size = -1
				it.type = tLookup.fromTypeSystem(ITypeSystem.ARRAY)
				it.typeArguments += innerTs
			]
			aD--
		}
		ts
	}

	def property(IField field) {
		factory.createProperty => [
			name = field.elementName;
			typeSpecifier = buildArrayOrRegularTypeSpecifier(tLookup.resolveType(field), tLookup.arrayDepth(field))
			typeSpecifier.typeArguments += tLookup.getTypeArguments(field).map[toTypeSpecifier]
			const = field.isIFieldConst
			static = field.isIFieldStatic
			eAdapters.add(new JavaFileLocationAdapter(field.declaringType.fullyQualifiedName, field.nameRange))
		]
	}

	def property(IMethod method) {
		factory.createProperty => [
			name = method.elementName;
			typeSpecifier = buildArrayOrRegularTypeSpecifier(tLookup.resolveType(method), tLookup.arrayDepth(method))
			eAdapters.add(new JavaFileLocationAdapter(method.declaringType.fullyQualifiedName, method.nameRange))
		]
	}

	def createEnumerator(IField enumerator) {
		factory.createEnumerator => [
			
			name = enumerator.elementName
			if(enumConstValues.get(name) !== null){
				
				val primitiveValue = enumConstValues.get(name)._value
			
				metaFeatures += _variable(name,primitiveValue.infer.type,primitiveValue) => [
					const = true
				]
				
			}
			eAdapters.add(
				new JavaFileLocationAdapter(enumerator.declaringType.fullyQualifiedName, enumerator.nameRange))
		]
	}

	def isITypeAbstract(IType type) {
		if(type.interface) return true
		Flags.isAbstract(type.flags)
	}

	def isIFieldConst(IField field) {
		Flags.isFinal(field.flags) || field.declaringType.isInterface
	}
	
	def isIFieldStatic(IField field) {
		Flags.isStatic(field.flags) || field.declaringType.isInterface
	}

	def setURI(URI uri) {
		this.uri = uri
		tLookup.URI = uri
	}

	def TypeSpecifier toTypeSpecifier(Type type) {
		val ts = factory.createTypeSpecifier
		ts.type = type
		ts
	}
}
