/** 
 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Contributors:
 * andreas muelder - itemis AG
 */
package com.yakindu.sct.domain.java.typesystem

import com.yakindu.base.types.Type
import com.yakindu.base.types.typesystem.GenericTypeSystem
import com.google.common.collect.BiMap
import com.google.common.collect.HashBiMap
import com.google.inject.Singleton

/** 
 * @author andreas muelder - Initial contribution and API
 */
@Singleton
class JavaTypeSystem extends GenericTypeSystem {
	
	public static final String BYTE = "byte"
	public static final String CHARACTER = "char"
	public static final String DOUBLE = "double"
	public static final String FLOAT = "float"
	public static final String INT = "int"
	public static final String LONG = "long"
	public static final String SHORT = "short"
	public static final String VOID = "void"
	public static final String NULL = "null"
	public static final String CONSTRUCTOR = "new"
	
	static final JavaTypeSystem INSTANCE = new JavaTypeSystem()

	BiMap<String, String> boxingMap

	def static JavaTypeSystem getInstance() {
		return INSTANCE
	}

	override protected void initRegistries() {
		super.initRegistries()
		getType(REAL).setAbstract(true)
		getType(INTEGER).setAbstract(true)
		getType(STRING).setAbstract(true)
		
		getType(ARRAY).setAbstract(false)
		
		declarePrimitive(BYTE)
		declarePrimitive(CHARACTER)
		declarePrimitive(DOUBLE)
		declarePrimitive(FLOAT)
		declarePrimitive(INT)
		declarePrimitive(LONG)
		declarePrimitive(SHORT)
		declarePrimitive(VOID)
		declarePrimitive(NULL)
		declareSuperType(getType(DOUBLE), getType(REAL))
		declareSuperType(getType(FLOAT), getType(DOUBLE))
		// make integers assignable to floats and doubles
		declareSuperType(getType(INTEGER), getType(FLOAT))
		// long > int > short > byte
		declareSuperType(getType(LONG), getType(INTEGER))
		declareSuperType(getType(INT), getType(LONG))
		declareSuperType(getType(SHORT), getType(INT))
		declareSuperType(getType(BYTE), getType(SHORT))
		enableAutoBoxing()
	}

	def enableAutoBoxing() {
		boxingMap = HashBiMap.create()
		declareBoxConversion(Boolean.getName(), BOOLEAN)
		declareBoxConversion(Byte.getName(), BYTE)
		declareBoxConversion(Character.getName(), CHARACTER)
		declareBoxConversion(Double.getName(), DOUBLE)
		declareBoxConversion(Float.getName(), FLOAT)
		declareBoxConversion(Integer.getName(), INT)
		declareBoxConversion(Long.getName(), LONG)
		declareBoxConversion(Short.getName(), SHORT)
		declareBoxConversion(String.getName(), STRING)
	}

	def protected declareBoxConversion(String type1, String type2) {
		boxingMap.put(type1, type2)
	}

	override boolean isConvertableTo(Type type1, Type type2) {
		return isBoxConvertableTo(type1, type2) || super.isConvertableTo(type1, type2)
	}

	override protected Type getConversionType(Type sourceType) {
		var Type convType = super.getConversionType(sourceType)
		if (convType === null) {
			return getType(boxingMap.get(sourceType.getId()))
		}
		return null
	}

	def protected boolean isBoxConvertableTo(Type type1, Type type2) {
		if (type1?.getId() === null || type2?.getId() === null) {
			return false
		}
		return isBoxConvertableToSubType(type1, type2) || isBoxConvertableToSubType(type2, type1)
	}

	def protected boolean isBoxConvertableToSubType(Type type1, Type type2) {
		var String conversionTypeName = boxingMap.get(type1.getId())
		var Type conversionType = getType(conversionTypeName)
		if (conversionType !== null) {
			if (isSuperType(conversionType, type2)) {
				return true
			}
		}
		return type2.getId().equals(conversionTypeName)
	}
}
