/**
 * Copyright (c) 2020 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.model.stext.validation

import com.google.inject.Inject
import com.yakindu.sct.model.sgraph.Entry
import com.yakindu.sct.model.sgraph.Exit
import com.yakindu.sct.model.sgraph.ReactionProperty
import com.yakindu.sct.model.sgraph.Region
import com.yakindu.sct.model.sgraph.Transition
import com.yakindu.sct.model.sgraph.Vertex
import com.yakindu.sct.model.stext.concepts.EntryTransition
import com.yakindu.sct.model.stext.concepts.ExitTransition
import com.yakindu.sct.model.stext.stext.EntryPointSpec
import com.yakindu.sct.model.stext.stext.ExitPointSpec
import java.util.Iterator
import java.util.List
import java.util.Map
import org.eclipse.emf.common.util.EList
import org.eclipse.xtext.validation.Check
import org.eclipse.xtext.validation.CheckType
import com.yakindu.sct.model.sgraph.util.SgraphExtensions

class EntryExitPointValidator extends STextBaseValidator {
	
	@Inject protected extension ExitTransition
	@Inject protected extension EntryTransition
	@Inject protected extension SgraphExtensions

	public static final String EXITPOINTSPEC_WITH_TRIGGER = "Transition with an exit point spec does not have a trigger or guard.";

	@Check(CheckType.FAST)
	def void checkExitPointSpecWithTrigger(Transition it) {
		if (!properties.filter(ExitPointSpec).isEmpty && trigger !== null &&
			source instanceof com.yakindu.sct.model.sgraph.State) {
			error(EXITPOINTSPEC_WITH_TRIGGER, it, null, -1)
		}
	}

	public static final String ONLY_FIRST_ENTRY_POINT_WILL_BE_USED = "Ignoring additional entry points. Using only the first: '%s'";

	@Check(CheckType.FAST)
	def void checkOnlyOneEntryPointSpecIsUsed(Transition transition) {
		if (transition.properties.filter(EntryPointSpec).size > 1) {
			warning(String.format(ONLY_FIRST_ENTRY_POINT_WILL_BE_USED, transition.properties.filter(EntryPointSpec).head.entrypoint), transition, null, -1)
		}
	}

	public static final String ENTRY_UNUSED = "The named entry '%s' is not used by incoming transitions.";
	public static final String ENTRY_NOT_EXIST = "The named entry does not exist: ";

	@Check(CheckType.FAST)
	def void checkUnusedEntry(Entry entry) {
		
		if (entry.parentRegion.composite instanceof com.yakindu.sct.model.sgraph.State &&
			entry.incomingTransitions.isEmpty) {
			var com.yakindu.sct.model.sgraph.State state = (entry.parentRegion.
				composite as com.yakindu.sct.model.sgraph.State)
			if (!entry.isDefault()) {
				if (state.incomingTransitions
					.map[properties].flatten
					.filter(EntryPointSpec)
					.filter(p | entry.name.equals(p.entrypoint))
					.empty
				) {
					warning(String.format(ENTRY_UNUSED, entry.name) , entry, null, -1)
				}
			}
		}
	}

	public static final String TRANSITION_UNBOUND_DEFAULT_ENTRY_POINT = "Target state '%s' has regions without 'default' entries.";
	public static final String TRANSITION_UNBOUND_NAMED_ENTRY_POINT = "Target state '%s' has regions without named entries: ";

	public static final String REGION_UNBOUND_DEFAULT_ENTRY_POINT = "Region%s must have a 'default' entry.";
	public static final String REGION_UNBOUND_NAMED_ENTRY_POINT = "Region%s should have a named entry to support transitions entry specification: ";

	@Check(CheckType.NORMAL)
	def void checkUnboundEntryPoints(com.yakindu.sct.model.sgraph.State state) {
		if (state.isComposite()) {
			val List<List<Transition>> transitions = state.incomingTransitions.entrySpecSortedTransitions
			var Map<Region, List<Entry>> regions = null
			// first list contains Transitions without entry spec
			if (!transitions.get(0).isEmpty()) {
				regions = state.regions.getRegionsWithoutDefaultEntry
				if (!regions.isEmpty()) {
					for (Transition transition : transitions.get(0)) {
						error(String.format(TRANSITION_UNBOUND_DEFAULT_ENTRY_POINT, transition.target.name), transition, null, -1)
					}
					for (Region region : regions.keySet()) {
						error(String.format(REGION_UNBOUND_DEFAULT_ENTRY_POINT, region.name === null ? "" : (" '" + region.name + "'")), region, null, -1)
					}
				}
			}
			// second list contains Transitions with entry spec
			if (!transitions.get(1).isEmpty()) {
				if (regions === null) {
					regions = state.regions.getRegionsWithoutDefaultEntry
				}
				for (Transition transition : transitions.get(1)) {
					var boolean hasTargetEntry = true
					for (ReactionProperty property : transition.getProperties()) {
						if (property instanceof EntryPointSpec) {
							var EntryPointSpec spec = property
							var String specName = ''''«»«spec.getEntrypoint()»'«»'''
							for (Region region : regions.keySet()) {
								var boolean hasEntry = false
								for (Entry entry : regions.get(region)) {
									if (entry.getName().equals(spec.getEntrypoint())) {
										hasEntry = true
									// TODO cancel here
									}
								}
								if (!hasEntry) {
									error(String.format(REGION_UNBOUND_NAMED_ENTRY_POINT + specName, region.name === null ? "" : (" '" + region.name + "'")), region, null, -1)
									hasTargetEntry = false
								}
							}
							if (!hasTargetEntry) {
								error(String.format(TRANSITION_UNBOUND_NAMED_ENTRY_POINT + specName, transition.target.name), transition, null, -1)
							}
							var boolean usingEntry = false
							for (Region region : state.regions) {
								var EList<Vertex> vertices = region.vertices
								for (Vertex vertice : vertices) {
									if (vertice instanceof Entry) {
										if (spec.getEntrypoint().equals(vertice.getName())) {
											usingEntry = true
										}
									}
								}
							}
							if (!usingEntry) {
								warning(ENTRY_NOT_EXIST + specName, transition, null, -1)
							}
						}
					}
				}
			}
		}
	}
	

	public static final String EXIT_UNUSED = "Exit%s is not connected to any outgoing transitions.";
	public static final String EXIT_NEVER_USED = "The named exit is not used: ";

	@Check(CheckType.NORMAL)
	def void checkUnusedExit(Exit exit) {
		if (exit.getParentRegion().getComposite() instanceof com.yakindu.sct.model.sgraph.State &&
			exit.outgoingTransitions.isEmpty()) {
			var com.yakindu.sct.model.sgraph.State state = (exit.getParentRegion().
				getComposite() as com.yakindu.sct.model.sgraph.State)
			if (!exit.isDefault()) {
				var boolean hasOutgoingTransition = false
				var boolean equalsOutgoingTransition = false
				var Iterator<Transition> transitionIt = state.outgoingTransitions.iterator()
				while (transitionIt.hasNext() && !hasOutgoingTransition) {
					var Transition transition = transitionIt.next()
					hasOutgoingTransition = transition.isDefaultExitTransition ||
						transition.isNamedExitTransition(exit.getName())
				}
				if (!hasOutgoingTransition) {
					val exName = exit.name.blank ? "" : (" '" + exit.name + "'")
					error(String.format(EXIT_UNUSED, exName), exit, null, -1)
				}
				for (Transition transition : state.outgoingTransitions) {
					for (ReactionProperty property : transition.getProperties()) {
						if (property instanceof ExitPointSpec) {
							var String exitpoint = property.getExitpoint()
							if (exitpoint.equals(exit.getName())) {
								equalsOutgoingTransition = true
							}
						}
					}
				}
				if (!equalsOutgoingTransition) {
					warning('''«EXIT_NEVER_USED»'«»«exit.getName()»'«»''', exit, null, -1)
				}
			} else {
				var boolean hasOutgoingTransition = false
				var Iterator<Transition> transitionIt = state.outgoingTransitions.iterator()
				while (transitionIt.hasNext() && !hasOutgoingTransition) {
					hasOutgoingTransition = transitionIt.next().isDefaultExitTransition
				}
				if (!hasOutgoingTransition) {
					val exName = exit.name.blank ? "" : (" '" + exit.name + "'")
					error(String.format(EXIT_UNUSED, exName), exit, null, -1)
				}
			}
		}
	}

}
