/**
 * Copyright (c) 2020-2024 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */

package com.yakindu.base.expressions.interpreter.base

/**
 * Provides executions for expressions.
 * 
 * @author axel terfloth
 */

import com.google.inject.Inject
import com.yakindu.base.base.NamedElement
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.expressions.AssignmentExpression
import com.yakindu.base.expressions.expressions.AssignmentOperator
import com.yakindu.base.expressions.expressions.BinaryExpression
import com.yakindu.base.expressions.expressions.BitwiseAndExpression
import com.yakindu.base.expressions.expressions.BitwiseOrExpression
import com.yakindu.base.expressions.expressions.BitwiseXorExpression
import com.yakindu.base.expressions.expressions.BlockExpression
import com.yakindu.base.expressions.expressions.BoolLiteral
import com.yakindu.base.expressions.expressions.ConditionalExpression
import com.yakindu.base.expressions.expressions.DoubleLiteral
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.EventRaisingExpression
import com.yakindu.base.expressions.expressions.EventValueReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.expressions.FloatLiteral
import com.yakindu.base.expressions.expressions.IfExpression
import com.yakindu.base.expressions.expressions.InitializationExpression
import com.yakindu.base.expressions.expressions.IntLiteral
import com.yakindu.base.expressions.expressions.LogicalAndExpression
import com.yakindu.base.expressions.expressions.LogicalNotExpression
import com.yakindu.base.expressions.expressions.LogicalOrExpression
import com.yakindu.base.expressions.expressions.MetaCall
import com.yakindu.base.expressions.expressions.NullLiteral
import com.yakindu.base.expressions.expressions.NumericalUnaryExpression
import com.yakindu.base.expressions.expressions.ParenthesizedExpression
import com.yakindu.base.expressions.expressions.PostFixUnaryExpression
import com.yakindu.base.expressions.expressions.PrimitiveValueExpression
import com.yakindu.base.expressions.expressions.ReturnExpression
import com.yakindu.base.expressions.expressions.StringLiteral
import com.yakindu.base.expressions.expressions.TypeCastExpression
import com.yakindu.base.expressions.interpreter.CoreFunction
import com.yakindu.base.expressions.interpreter.context.IExecutionSlotResolver
import com.yakindu.base.types.Argument
import com.yakindu.base.types.EnumerationType
import com.yakindu.base.types.Enumerator
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Property
import com.yakindu.base.types.Type
import com.yakindu.base.types.TypesUtil
import com.yakindu.base.types.inferrer.ITypeSystemInferrer
import com.yakindu.base.types.inferrer.ITypeSystemInferrer.InferenceResult
import com.yakindu.base.types.typesystem.GenericTypeSystem
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.base.types.util.ArgumentSorter
import com.yakindu.sct.model.sruntime.CompositeExecutionEvent
import com.yakindu.sct.model.sruntime.CompositeSlot
import com.yakindu.sct.model.sruntime.ExecutionEvent
import com.yakindu.sct.model.sruntime.ExecutionSlot
import com.yakindu.sct.model.sruntime.ReferenceExecutionEvent
import com.yakindu.sct.model.sruntime.SRuntimeFactory
import java.util.ArrayDeque
import java.util.List
import java.util.Map
import org.eclipse.emf.ecore.util.EcoreUtil
import org.eclipse.xtext.naming.IQualifiedNameProvider

import static com.yakindu.base.expressions.interpreter.base.IInterpreter.*

class ExpressionExecution extends BaseExecution {

	protected static Map<AssignmentOperator, String> assignFunctionMap = #{
		AssignmentOperator.MULT_ASSIGN -> "*",
		AssignmentOperator.DIV_ASSIGN -> "/",
		AssignmentOperator.MOD_ASSIGN -> "%",
		AssignmentOperator.ADD_ASSIGN -> "+",
		AssignmentOperator.SUB_ASSIGN -> "-",
		AssignmentOperator.LEFT_SHIFT_ASSIGN -> "<<",
		AssignmentOperator.RIGHT_SHIFT_ASSIGN -> ">>",
		AssignmentOperator.AND_ASSIGN -> "&",
		AssignmentOperator.XOR_ASSIGN -> "^",
		AssignmentOperator.OR_ASSIGN -> "|"
	};

	@Inject protected extension SRuntimeFunction cf
	@Inject protected extension IQualifiedNameProvider qnp
	@Inject protected extension ITypeSystem ts
	@Inject protected extension ITypeSystemInferrer tsi
	@Inject protected extension IExecutionSlotResolver
	@Inject protected extension TypesUtil
	protected extension SRuntimeFactory runtimeFactory = SRuntimeFactory.eINSTANCE
	@Inject protected IInstanceFactory factory

	@Inject protected extension IInstanceFactory

	override provideExecution(Object program) {
		program.execution
	}

	def dispatch void execution(Object it) {
	}

	def dispatch void execution(BlockExpression block) {
		block.expressions.forEach[_exec]
	}

	def dispatch void execution(InitializationExpression expr) {
		val typeInfo = expr.typeInfoUsingArraySize(expr.arguments.size)
		
		expr.arguments.forEach [ arg |
			arg.value._exec
			_value
		]
		if (typeInfo.type.isMap) {
			expr.arguments.forEach [ arg |
				arg.key._exec
				_value
			]
		}
		
		_return [
			
			val argKeys = new ArrayDeque<Object>
			if (typeInfo.type.isMap) {
				for (i : 1 .. expr.arguments.size) {
					argKeys.push(popValue)
				}
			}
			
			val argValues = new ArrayDeque<Object>
			for (i : 1 .. expr.arguments.size) {
				argValues.push(popValue)
			}
			

			val instance = typeInfo.newInstance as CompositeSlot
			if (typeInfo.type.isMap) {
				expr.arguments.forEach [ arg, i |
					val key = argKeys.get(i)
					val name = key.toString
					val value = argValues.get(i)
					var slot = instance.slots.findFirst[s|s.key == key]
					if (slot === null) {
						val newMapEntry = (
							if ( value instanceof ExecutionSlot) factory.copyInstance(value) 
							else factory.newInstance(instance.typeSpecifier.typeArguments.get(1))
						) as ExecutionSlot => [ s |
							s.name = name
							s.fqName = name
							s.key = key
							
							if (!(value instanceof ExecutionSlot))
								s.value = cast(value, s.type)
						]
						
						instance.slots += newMapEntry
					}
				]
			} else {
				argValues.forEach [ v, i |
					val slot = instance.slots.get(i)
					set(slot, cast(v, (slot as ExecutionSlot).type))
				]
			}
			return instance
		]
	}

	def protected InferenceResult typeInfoUsingArraySize(Expression expr, int size) {
		var typeInfo = expr.infer
		if (typeInfo.type.isArray)
			typeInfo = InferenceResult.from(typeInfo.type, typeInfo.bindings, size)
		return	typeInfo
	}
	
	
	def dispatch eventValue(ExecutionEvent it) {
		it.value
	}

	def dispatch eventValue(CompositeExecutionEvent it) {
		EcoreUtil.copy(it) => [
			it.name = it.name + "@value"
		]
	}

	def dispatch eventValue(ReferenceExecutionEvent it) {
		EcoreUtil.copy(it) => [
			it.name = it.name + "@value"
		]
	}

	def dispatch void execution(InitializationExpression context, Operation item) {
		_try("call " + item.name, [
			item.executeArguments(context.arguments)
			if (context instanceof FeatureCall) {
				context.owner._exec
				_execute[
					val slot = popValue
					slot.instance.invoke(slot, context)
				]
			} else {
				context._call(null, item)
			}
		], [ onError |
			doResolve(item, context)
		])
	}

	def dispatch void execution(PrimitiveValueExpression expr) {
		_return(expr.value.literalValue.toString, [
			expr.value.literalValue
		])
	}

	def dispatch void execution(ParenthesizedExpression expr) {
		expr.expression._exec
	}

	def dispatch void execution(AssignmentExpression expr) {
		expr.expression._exec
		_value
		expr.varRef._exec
		_return(expr.operator.literal, [
			val f = assignFunctionMap.get(expr.operator)
			val varRef = popValue
			var value = popValue

			if (expr.operator !== AssignmentOperator.ASSIGN) {
				val varValue = varRef.get
				value = evaluate(f, varValue, value)
			}

			varRef.set = cast(value, (varRef as ExecutionSlot).type)

			return value
		])
	}

	def dispatch void execution(ElementReferenceExpression expr) {
		expr.reference.execution(expr)
	}

	def dispatch void execution(FeatureCall expr) {
		expr.feature.execution(expr)
	}

	def dispatch void execution(Object item, Object context) {
		throw new IllegalArgumentException('''Don't know how to execute «item.class.simpleName» in context «context.class .simpleName»''')
	}

	def dispatch void execution(Operation item, ArgumentExpression context) {
		_try("call " + item.name, [
			item.executeArguments(context.arguments)
			if(context instanceof FeatureCall && !item.name.equals("new")){
				(context as FeatureCall).owner._exec
				_execute[
					val slot = popValue
					slot.instance.invoke(slot, context)
				]
			} else {
				
				context._call(null, item)
			}
		], [ onError |
			doResolve(item, context)
		])
	}

	def dispatch void execution(NamedElement item, ArgumentExpression context) {
		doResolve(item, context)
	}

	protected def dispatch void doResolve(NamedElement item, ElementReferenceExpression context) {
		_return('''resolve «item.symbol»''', [
			executeResolve(item, context)
		])
		if ((context.isArrayAccess && context.reference?.infer?.type.isMap)) {
			resolveMapElement(context.arraySelector)

		} else if (context.isArrayAccess) {
			resolveArrayElement(context.arraySelector)

		}
	}

	protected def dispatch void doResolve(NamedElement item, FeatureCall context) {
		context.owner._exec
		_return('''resolve «item.symbol»''', [
			executeResolve(item, context)
		])
		// TODO: Map access via feature call
		if (context.isArrayAccess) {
			resolveArrayElement(context.arraySelector)
		}
	}

	protected def dispatch void doResolve(Enumerator item, ArgumentExpression context) {
		_return('''resolve «item.symbol»''', [
			item
		])
	}

	protected def dispatch void doResolve(InitializationExpression item, EventRaisingExpression context) {
		_return('''resolve «context.event.symbol»''', [
			resolve(null, (context.event as ElementReferenceExpression).reference.symbol)
		])
	}

	protected def dispatch void doResolve(Enumerator item, InitializationExpression context) {
		_return('''resolve «item.symbol»''', [
			item.literalValue as long
		])
	}

	protected def dispatch executeResolve(NamedElement item, ElementReferenceExpression context) {
		resolve(null, item.symbol)
	}

	protected def dispatch executeResolve(NamedElement item, FeatureCall context) { popValue.resolve(item.symbol) }

	protected def resolveArrayElement(List<Expression> arraySelectors) {
		arraySelectors.head._exec
		_value
		_return [
			val arrayIndex = popValue
			val arraySlot = popValue
			arraySlot.resolve(arrayIndex)
		]
	}

	protected def resolveMapElement(List<Expression> arraySelectors) {
		arraySelectors.head._exec
		_value
		_return [
			val key = popValue
			val variable = popValue as CompositeSlot
			var slot = variable.slots.findFirst[ s | s.key == key]
			
			if (slot === null) {
				// Implicitly create a new map entry
				val newMapEntry = factory.newInstance(variable.typeSpecifier.typeArguments.get(1)) as ExecutionSlot => [
					it.name = key.toString
					it.fqName = it.name
					it.key = key
				]
				variable.slots += newMapEntry
				newMapEntry
			} else {
				slot
			}
		]
	}

	def protected void executeArguments(Operation op, List<Argument> args) {
		ArgumentSorter.getOrderedExpressions(args, op).forEach [ a |
			a.get._exec
			_value
		]
	}

	def dispatch void execution(MetaCall expr) {

		val metaFeature = expr.feature
		if (metaFeature instanceof Property) {
			if (metaFeature.name == "value") {
				expr.owner._exec
				_return[
					val eventSlot = popValue()
					(eventSlot as ExecutionEvent).eventValue
				]
				return
			}
		}
		throw new IllegalArgumentException("Cannot resolve meta call '" + expr + "'")
	}

	def dispatch String symbol(Object it) {
		null
	}

	def dispatch String symbol(NamedElement it) {
		it.fullyQualifiedName.lastSegment
	}

	def dispatch void execution(BinaryExpression it) {
		binaryExpressionExecution(operator.getName, leftOperand, rightOperand)
	}

	protected def void binaryExpressionExecution(String operator, Expression left, Expression right) {
		left._exec
		_value
		right._exec
		_value
		_return(operator, [
			val rightValue = popValue
			val leftValue = popValue

			operator.evaluate(leftValue, rightValue)
		])
	}

	protected def void unaryExpressionExecution(String operator, Expression expr) {
		expr._exec
		_value
		_return(operator, [
			val value = popValue

			evaluate(operator, value)
		])
	}

	def dispatch void execution(LogicalOrExpression expr) {
		expr.leftOperand._exec
		_value
		_execute('||', [
			if (popValue == true) {
				_return[true]
			} else {
				expr.rightOperand._exec
				_value
				_return[popValue]
			}
		])
	}

	def dispatch void execution(LogicalAndExpression expr) {
		expr.leftOperand._exec
		_value
		_execute('&&', [
			if (popValue == false) {
				_return[false]
			} else {
				expr.rightOperand._exec
				_value
				_return[popValue]
			}
		])
	}

	def dispatch void execution(LogicalNotExpression it) {
		operand._exec
		_value
		_return("!", [
			popValue == false
		])
	}

	def dispatch void execution(ConditionalExpression it) {
		condition._exec
		_value
		_execute[
			if (popValue == true) {
				trueCase._exec
			} else {
				falseCase._exec
			}
		]
	}

	def dispatch void execution(EventRaisingExpression it) {

		it.event._exec
		if (it.value !== null) {
			it.value._exec
			_value
		}

		_execute[
			val value = if(it.value !== null) popValue else null
			val slot = popValue
			raise(slot, value)
		]
	}

	def dispatch void execution(EventValueReferenceExpression expression) {

		expression.value._exec
		_return[
			val eventSlot = popValue()
			(eventSlot as ExecutionEvent).eventValue
		]
	}

	def dispatch void execution(BitwiseAndExpression it) {
		binaryExpressionExecution(CoreFunction.BIT_AND, leftOperand, rightOperand)
	}

	def dispatch void execution(BitwiseOrExpression it) {
		binaryExpressionExecution(CoreFunction.BIT_OR, leftOperand, rightOperand)
	}

	def dispatch void execution(BitwiseXorExpression it) {
		binaryExpressionExecution(CoreFunction.BIT_XOR, leftOperand, rightOperand)
	}

	def dispatch void execution(NumericalUnaryExpression expression) {
		unaryExpressionExecution(expression.operator.literal, expression.operand)
	}

	def dispatch void execution(PostFixUnaryExpression it) {
		operand._exec
		_return(it.operator.literal, [
			val varRef = popValue
			val varValue = varRef.get
			val opValue = evaluate(operator.literal, varValue)
			varRef.set(opValue)
			varValue
		])
	}

	def dispatch void execution(TypeCastExpression expression) {
		expression.operand._exec
		_value
		_return("cast(" + expression.type.originType.name + ")", [
			val operandValue = popValue
			typeCast(operandValue, expression.type.originType)
		])
	}

	def dispatch void execution(ReturnExpression it) {
		if (it.expression !== null) {
			it.expression._exec
			_value
		}
		_execute("return expression", [
			exitCall(if(it.expression === null) null else popValue)
		])
	}

	def dispatch void execution(IfExpression it) {
		condition._exec
		_value
		_execute('if', [
			val checkResult = (popValue == true)
			if(checkResult) then._exec else if(^else !== null) ^else._exec
		])
	}

	def dispatch void execution(Operation it) {
		_delegate
	}

	def Object cast(Object value, Type type) {
		if (type !== null) {
			typeCast(value, type.originType)
		} else {
			value
		}
	}

	def protected dispatch Object typeCast(Long value, Type type) {
		if(type instanceof EnumerationType) return value
		if(ts.isSuperType(type, ts.getType(GenericTypeSystem.INTEGER))) return value
		if(ts.isSuperType(type, ts.getType(GenericTypeSystem.REAL))) return Double.valueOf(value)
		throw new InterpreterException("Invalid cast from Long to " + type.name)
	}

	def protected dispatch Object typeCast(Float value, Type type) {
		if(ts.isSuperType(type, ts.getType(GenericTypeSystem.INTEGER))) return value.longValue
		if(ts.isSuperType(type, ts.getType(GenericTypeSystem.REAL))) return Double.valueOf(value)
		throw new InterpreterException("Invalid cast from Float to " + type.name)
	}

	def protected dispatch Object typeCast(Double value, Type type) {
		if(ts.isSuperType(type, ts.getType(ITypeSystem.INTEGER))) return value.longValue
		if(ts.isSuperType(type, ts.getType(ITypeSystem.REAL))) return Double.valueOf(value)
		throw new InterpreterException("Invalid cast from Double to " + type.name)
	}

	def protected dispatch Object typeCast(Boolean value, Type type) {
		if(ts.isSuperType(type, ts.getType(ITypeSystem.BOOLEAN))) return value
		throw new InterpreterException("Invalid cast from Boolean to " + type.name)
	}

	def protected dispatch Object typeCast(String value, Type type) {
		if(ts.isSuperType(type, ts.getType(ITypeSystem.STRING))) return value
		throw new InterpreterException("Invalid cast from String to " + type.name)
	}

	def protected dispatch Object typeCast(Enumerator value, Type type) {
		if(ts.isSuperType(type, value.owningEnumeration)) return value
		throw new InterpreterException("Invalid cast from Enumerator to " + type.name)
	}

	def protected dispatch Object typeCast(Object value, Type type) {
		value
	}

	def dispatch literalValue(IntLiteral literal) {
		return literal.value as long
	}

	def dispatch literalValue(BoolLiteral bool) {
		return bool.value
	}

	def dispatch literalValue(DoubleLiteral literal) {
		return literal.value
	}

	def dispatch literalValue(FloatLiteral literal) {
		return literal.value
	}

	def dispatch literalValue(StringLiteral literal) {
		return literal.value
	}

	def dispatch literalValue(NullLiteral literal) {
		return NULL
	}

	def void _call(Object caller, Object receiver, Object operation) {
		executionCall(new Invokation(caller, receiver, operation))
	}

	def dispatch void executionCall(Object it) {
		throw new UnsupportedOperationException("Don't know how to call operations")
	}

	def dispatch void executionCall(Invokation it) {
		_delegate
	}

	def dispatch void executionCall(Operation it) {
		_delegate
	}

}
