/**
 * Copyright (c) 2024 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.base.expressions.interpreter.types

import com.google.inject.Inject
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.interpreter.base.IInstanceFactory
import com.yakindu.base.expressions.interpreter.base.IInterpreter
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Package
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.sct.model.sruntime.CompositeSlot
import com.yakindu.sct.model.sruntime.ExecutionSlot
import com.yakindu.sct.model.sruntime.SRuntimeFactory

/**
 * Instance behavior implementation for maps.
 * 
 * @author Andreas Mülder
 */
class MapInstance extends RuntimeInstance {

	@Inject extension ITypeSystem
	protected extension SRuntimeFactory runtimeFactory = SRuntimeFactory.eINSTANCE

	protected ComplexType mapType
	protected Package declarationPackage

	override void setUp(CompositeSlot instance, IInterpreter.Context context, IInstanceFactory factory) {
		super.setUp(instance, context, factory)
		this.mapType = ITypeSystem.MAP.type as ComplexType
		this.declarationPackage = this.mapType.eContainer as Package
	}

	override declarationPackage() {
		declarationPackage
	}

	override declarations() {
		mapType.allFeatures
	}

	/**
	 * put(k:KeyType, v:ValueType):ValueType // returns previous value if overwritten, else null
	 * remove(k:KeyType): void
	 * size(): int
	 * containsKey(k:KeyType): boolean
	 * containsValue(v:ValueType): boolean
	 * keys(): KeyType[]
	 * values(): ValueType[]
	 * clear(): void
	 */
	override dispatch void doInvoke(ArgumentExpression expr) {
		val feature = expr.featureOrReference

		switch (feature) {
			Operation case "put" == feature.name:
				put()
			Operation case "remove" == feature.name:
				remove()
			Operation case "size" == feature.name:
				size()
			Operation case "containsKey" == feature.name:
				containsKey()
			Operation case "containsValue" == feature.name:
				containsValue()
			Operation case "keys" == feature.name:
				keys()
			Operation case "values" == feature.name:
				values()
			Operation case "clear" == feature.name:
				clear()
			default:
				super._doInvoke(expr)
		}
	}

	def protected put() {
		val value = popValue
		val key = popValue
		val existingSlot = instanceSlot.resolveSlot(key)
		if (existingSlot === null) {
			val newSlot = SRuntimeFactory.eINSTANCE.createExecutionVariable => [ slot |
				slot.value = value
				slot.name = "" + key
				slot.key = key
				slot.type = getType(ITypeSystem.ANY) //TODO calculate correct type
			]
			instanceSlot.slots += newSlot
			pushValue(IInterpreter.NULL)
		} else {
			val oldValue = existingSlot.get
			existingSlot.value = value
			pushValue(oldValue)
		}
	}

	def protected remove() {
		val value = popValue
		val slotValue = instanceSlot.slots.findFirst[slot|slot.key == value]
		if (slotValue !== null)
			instanceSlot.slots.remove(slotValue)
	}

	def protected size() {
		pushValue(instanceSlot.slots.size)
	}

	def protected containsKey() {
		val key = popValue
		pushValue(instanceSlot.slots.exists[slot|slot.key == key])
	}

	def protected containsValue() {
		val value = popValue
		pushValue(instanceSlot.slots.exists[slot|slot.value == value])
	}

	def protected clear() {
		instanceSlot.slots.clear
	}

	def protected keys() {
		val result = createCompositeSlot => [
			name = "result"
			type = getType(ITypeSystem.ANY) //TODO correct type
		]
		instanceSlot.slots.forEach [ slot |
			val copy = factory.copyInstance(slot) as ExecutionSlot
			copy.value = copy.key
			result.slots += copy
		]
		pushValue(result)
	}

	def protected values() {
		val result = createCompositeSlot => [
			name = "result"
			type = getType(ITypeSystem.ANY) //TODO correct type
		]
		instanceSlot.slots.forEach [ slot, index |
			val copy = factory.copyInstance(slot) as ExecutionSlot
			copy.name = '''[«index»]'''
			result.slots += copy
		]
		pushValue(result)
	}

	override set(Object slot, Object value) {
		if (slot === instanceSlot) {
			if (slot instanceof CompositeSlot) {
				if (value instanceof CompositeSlot) {
					slot.slots.clear
					value.slots.forEach [ s, i |
						slot.slots += (factory.copyInstance(s) as ExecutionSlot) => [
							name = s.name
							fqName = s.fqName
						]
					]
				}
			}
		} else {
			_valueSemantics.setValue(slot, value)
		}
	}

	override dispatch ExecutionSlot resolveSlot(CompositeSlot slot, String symbol) {
		slot.slotByKey(symbol)
	}

}
