/**
 * Copyright (c) 2024 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.base.expressions.interpreter.types

import com.google.inject.Inject
import com.google.inject.Injector
import com.google.inject.Singleton
import com.yakindu.base.expressions.interpreter.base.IInstanceFactory
import com.yakindu.base.expressions.interpreter.base.IInterpreter
import com.yakindu.base.expressions.interpreter.base.IInterpreter.Instance
import com.yakindu.base.expressions.interpreter.base.InstanceSemantics
import com.yakindu.base.expressions.interpreter.base.InterpreterException
import com.yakindu.base.expressions.interpreter.base.ResolvingRuntimeInstance
import com.yakindu.base.expressions.interpreter.base.ValueSemantics
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.Direction
import com.yakindu.base.types.EnumerationType
import com.yakindu.base.types.Event
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Package
import com.yakindu.base.types.Part
import com.yakindu.base.types.Property
import com.yakindu.base.types.Reaction
import com.yakindu.base.types.Type
import com.yakindu.base.types.TypeBuilder
import com.yakindu.base.types.TypeParameter
import com.yakindu.base.types.TypeSpecifier
import com.yakindu.base.types.annotations.ClassificationAnnotations
import com.yakindu.base.types.inferrer.ITypeSystemInferrer
import com.yakindu.base.types.inferrer.ITypeSystemInferrer.InferenceResult
import com.yakindu.base.types.typesystem.ITypeSystem
import com.yakindu.base.types.typesystem.ITypeValueProvider
import com.yakindu.sct.model.sruntime.CompositeSlot
import com.yakindu.sct.model.sruntime.ExecutionSlot
import com.yakindu.sct.model.sruntime.ReferenceSlot
import com.yakindu.sct.model.sruntime.SRuntimeFactory
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.util.EcoreUtil

import static com.yakindu.base.expressions.interpreter.types.RuntimeInstance.*
import com.yakindu.base.types.typesystem.ITypeSemantics
import com.yakindu.base.types.PrimitiveType

/**
 * This factory creates instances for type system elements like types and packages. 
 * 
 * @author axel terfloth
 */
@Singleton
class RuntimeInstanceFactory implements IInstanceFactory {

	protected extension SRuntimeFactory runtimeFactory = SRuntimeFactory.eINSTANCE

	//@Inject protected extension IInterpreter interpreter
	@Inject protected extension Injector injector
	@Inject protected extension ITypeSystemInferrer
	@Inject protected extension ITypeSemantics
	@Inject protected extension ValueSemantics
	@Inject protected extension InstanceSemantics
	@Inject protected extension PackageHelper
	@Inject protected extension TypeBuilder typeBuilder
	@Inject protected extension ClassificationAnnotations

	@Inject protected IInterpreter.Context instanceContext
	@Inject protected extension ITypeSystem ts
	@Inject protected ITypeValueProvider typeValueProvider

	protected long instanceCount = 0l

	protected def newInstanceId() {
		return instanceCount++
	}

	override newInstance(Object type) {
		type.createNewInstance
	}

	protected def dispatch Object createNewInstance(Object type) {
		throw new InterpreterException('''Don't know how to create new instance of type «type?.class.name» ! ''')
	}

	protected def dispatch Object createNewInstance(Event eventDecl) {
		eventDecl.createInstanceMember(heap as EObject) => [
			instanceContext.heap.slots += it
		]
	}

	protected def dispatch Object createNewInstance(Property varDecl) {
		varDecl.createInstanceMember(heap as EObject) => [
			it.fqName = it.name
			heap.slots += it
		]
	}

	protected def dispatch Object createNewInstance(Package pkg) {

		val pkgSlot = pkg.provideSlot

		if (pkgSlot.instance === null) {
			pkgSlot.adaptInstance(pkgSlot.createPackageInstance(pkg))

			pkg.createInstanceMembers(pkgSlot)

			pkg.importedPackages.forEach[createNewInstance]

			instanceContext._invoke(pkgSlot, INIT, #[]);
		}

		return pkgSlot
	}

	def protected CompositeSlot provideSlot(Package pkg) {
		pkg.name.split("\\.").fold(heap as CompositeSlot, [ s, c |
			var packageSlot = s.slots.filter(CompositeSlot).findFirst[slot|slot.name == c]
			if (packageSlot === null) {
				packageSlot = createCompositeSlot => [
					name = c
					fqName = if(s !== heap) s.fqName + '.' + c else c
				]
				s.slots.add(packageSlot)
			}
			packageSlot
		])
	}

	def protected Instance createPackageInstance(CompositeSlot instanceMemory, Package pkg) {
		new PackageInstance => [
			it.injectMembers
			it.setUp(instanceMemory, pkg, instanceContext, this)
		]
	}

	def protected createInstanceMembers(Package it, CompositeSlot slot) {
		it.member.filter[it instanceof Property || it instanceof Event].forEach [ m |
			val mSlot = m.createInstanceMember(it)
			if(mSlot !== null) slot.add(mSlot)
		]
	}

	protected def dispatch Object createNewInstance(ComplexType it) {
		createNewInstance(it._typeSpecifier)
	}

	protected def dispatch Object createNewInstance(TypeSpecifier it) {

		it.infer.createNewInstance
	}

	protected def dispatch Object createNewInstance(InferenceResult it) {

		if (it.type.isArray) {
			it.createNewArray
		} else if (type.isMap) {
			createNewMap
		} else {
			it.type.createNewInstance(it)
		}
	}

	protected def dispatch Object createNewInstance(Object type, TypeSpecifier ts) {
		throw new InterpreterException('''Don't know how to create new instance of type «type?.class.name» ! ''')
	}

	protected def dispatch Object createNewInstance(TypeParameter type, TypeSpecifier ts) {
		throw new InterpreterException('''Can't create instance for unbound type parameter «type?.name» ! ''')
	}

	protected def dispatch Object createNewInstance(ComplexType type, InferenceResult ts) {

		val rtInstance = createCompositeSlot => [
			it.name = '''«type.name»#«newInstanceId»'''
			it.fqName = it.name
			it.type = type
			it.visible = false
		]
		heap.slots += rtInstance

		rtInstance.adaptInstance(rtInstance.createComplexTypeInstance(type))

		type.createInstanceMembers(rtInstance)

		instanceContext._invoke(rtInstance, INIT, #[]);

		return rtInstance
	}

	protected def dispatch Object createNewInstance(PrimitiveType type, InferenceResult ts) {
		type.createExecutionVariable
	}

	protected def dispatch Object createNewInstance(EnumerationType type, InferenceResult ts) {
		type.createExecutionVariable
	}
	
	protected def createExecutionVariable(Type type) {
		createExecutionVariable => [
			it.name = '''«type.name»#«newInstanceId»'''
			it.fqName = it.name
			it.type = type
			it.value = type.defaultValue
			it.visible = true
		]
	}

	def protected Instance createComplexTypeInstance(CompositeSlot instanceMemory, ComplexType type) {
		new ComplexTypeInstance => [
			setUpInstance(instanceMemory, type)
		]
	} 
	
	def protected void setUpInstance(ComplexTypeInstance it, CompositeSlot instanceMemory, ComplexType type) {
			it.injectMembers
			it.setUp(instanceMemory, type, instanceContext, this)		
	}
	
	def protected createInstanceMembers(ComplexType it, CompositeSlot slot) {
		it.allFeatures.filter[it instanceof Property || it instanceof Event].forEach [ m |
			val mSlot = m.createInstanceMember(it) => [
				fqName = name
				visible = ! m.isSynthetic
			]
			if(mSlot !== null) slot.slots += mSlot
		]
	}

	def ExecutionSlot createInstanceMember(Declaration it, EObject parent) {
		it.doCreateInstanceMember(parent)
	}

	def dispatch ExecutionSlot doCreateInstanceMember(Declaration it, EObject parent) {
		throw new InterpreterException('''Don't know how to create new instance of type «it?.class.name» in «parent?.class.name»! ''')
	}

	def dispatch ExecutionSlot doCreateInstanceMember(Property prop, EObject parent) {
		createInstanceMemberByType(prop, prop.infer.type, parent)
	}

	def dispatch ExecutionSlot doCreateInstanceMember(Property prop, ComplexType parent) {
		createInstanceMemberByType(prop, prop.infer.type, parent)
	}

	def dispatch ExecutionSlot doCreateInstanceMember(Event event, EObject parent) {
		createInstanceMemberByType(event, event.type?.infer?.type, parent)
	}

	def dispatch ExecutionSlot doCreateInstanceMember(Reaction reaction, EObject parent) {
		null // create nothing for now
	}

	def dispatch ExecutionSlot doCreateInstanceMember(Operation op, EObject parent) {
		null // create nothing for now
	}

	def dispatch ExecutionSlot doCreateInstanceMember(EnumerationType prop, Package parent) {
		null
	}

	def dispatch ExecutionSlot doCreateInstanceMember(ComplexType prop, EObject parent) {
		null
	}

	protected def dispatch ExecutionSlot createInstanceMemberByType(Declaration decl, Type type, EObject parent) {
		throw new InterpreterException('''Don't know how to create new instance of type «type?.class.name» for «decl?.class.name» in «parent?.class.name»! ''')
	}

	protected def dispatch ExecutionSlot createInstanceMemberByType(Property prop, Type type, EObject parent) {
		type.createExecutionVariable => [ v |
			v.name = prop.name
			v.writable = !prop.const
		]
	}

	protected def dispatch ExecutionSlot createInstanceMemberByType(Property prop, EnumerationType type,
		EObject parent) {
		type.createExecutionVariable => [ v |
			v.name = prop.name
			v.writable = !prop.const
		]
	}

	protected def dispatch ExecutionSlot createInstanceMemberByType(Property prop, ComplexType type, EObject parent) {
		createReferenceSlot => [ s |
			s.name = prop.name
			s.type = type
			s.writable = !prop.const
			s.reference = null
		]

	}

	protected def dispatch ExecutionSlot createInstanceMemberByType(Part prop, ComplexType type, EObject parent) {
		prop.infer.createNewInstance as ExecutionSlot => [ s |
			s.name = prop.name
			s.type = type
			s.writable = !prop.const
		]
	}

	protected def dispatch ExecutionSlot createInstanceMemberByType(Event event, Type type, EObject parent) {	
		createExecutionEvent => [
			it.name = event.name
			it.type = event.type
			it.value = type.defaultValue
			it.direction = Direction.get(event.direction.value)
		]
	}

	protected def dispatch ExecutionSlot createInstanceMemberByType(Event event, Void type, EObject parent) {
		createExecutionEvent => [
			it.name = event.name
			it.type = event.type
			it.direction = Direction.get(event.direction.value)
		]
	}

	protected def dispatch ExecutionSlot createInstanceMemberByType(Event event, ComplexType type, EObject parent) {
		createReferenceExecutionEvent => [
			it.name = event.name
			it.type = type
			it.direction = Direction.get(event.direction.value)
			it.reference = null
		]
	}

	// ========================================================================
	// Handling arrays
	def protected createNewArray(InferenceResult inferenceResult) {

		val instance = createCompositeSlot => [
			it.typeSpecifier = inferenceResult.asTypeSpecifier
			it.name = '''«type.name»#«newInstanceId»'''
			it.fqName = it.name
			it.visible = false
			if(inferenceResult.arraySize.present && inferenceResult.arraySize.get > 0) slots +=
				createArraySlots(inferenceResult)
		]

		heap.slots += instance
		instance.adaptInstance(instance.createArrayInstance)

		instanceContext._invoke(instance, INIT, #[]);
		return instance
	}

	def protected Instance createArrayInstance(CompositeSlot instanceMemory) {
		new ArrayInstance => [
			it.injectMembers
			it.setUp(instanceMemory, instanceContext, this)
		]
	}

	/** TODO: instantiation should only use the factory itself. */
	def ExecutionSlot createSlotForCollectionElement(InferenceResult elementTypeSpecifier) {
		val elementType = elementTypeSpecifier.type

		switch (elementType) {
			EnumerationType:
				return createExecutionVariable => [
					value = elementType.defaultValue?.asValue
					writable = true
				]
			ComplexType case elementTypeSpecifier.type.isArray || elementType.isValueType:
				return elementTypeSpecifier.newInstance as ExecutionSlot
			ComplexType:
				createReferenceSlot => [ s |
					s.type = elementType
					s.writable = true
					s.reference = null
				]
			default:
				createExecutionVariable => [
					value = elementType.defaultValue?.asValue
					writable = true
				]
		}
	}

	def createArraySlots(InferenceResult specifier) {
		(0 .. specifier.arraySize.get - 1).map [ current |
			createSlotForCollectionElement(specifier.bindings.head) => [
				name = '''[«current»]'''
				type = specifier.bindings.head.type
			]
		]
	}

	// ========================================================================
	// Handling maps
	def protected createNewMap(InferenceResult inferenceResult) {

		val instance = createCompositeSlot => [
			it.typeSpecifier = inferenceResult.asTypeSpecifier
			it.name = '''«type.name»#«newInstanceId»'''
			it.fqName = it.name
		]

		heap.slots += instance
		instance.adaptInstance(instance.createMapInstance)

		instanceContext._invoke(instance, INIT, #[]);
		return instance
	}

	def protected Instance createMapInstance(CompositeSlot instanceMemory) {
		new MapInstance => [
			it.injectMembers
			it.setUp(instanceMemory, instanceContext, this)
		]
	}

	protected def defaultValue(Type t) {
		typeValueProvider.defaultValue(t)?.asValue
	}

	protected def add(CompositeSlot it, ExecutionSlot child) {
		it.slots.add(child)
		child.fqName = '''«it.fqName».«child.name»'''
	}

	def protected dispatch slots(Object it) {
		#[]
	}

	def protected dispatch slots(CompositeSlot it) {
		it.slots
	}

	// ===================================================================
	// CREATING instances by COPYING
	override copyInstance(Object instance) {
		instance.doCopy
	}

	protected def dispatch doCopy(Object instance) {
		throw new InterpreterException('''Don't know how to copy instance of type «instance?.class.name» ! ''')
	}

	/**
	 * Creates a copy of a CompositeSlot and also copies the associated instance if it exists.
	 */
	protected def dispatch doCopy(CompositeSlot slot) {
		EcoreUtil.copy(slot) => [
			it.name = slot.name + "@copy"

			if (slot.instance !== null) {
				slot.instance.cloneInstance(it)
			}
		]
	}

	/**
	 * Creates a copy of a ReferenceSlot.
	 */
	protected def dispatch doCopy(ReferenceSlot slot) {
		EcoreUtil.copy(slot) => [
			it.reference = slot.reference
		]
	}

	protected def dispatch doCopy(ExecutionSlot slot) {
		EcoreUtil.copy(slot)
	}

	protected def dispatch cloneInstance(ResolvingRuntimeInstance it, CompositeSlot slot) {
		val cloneInstance = it.clone as ResolvingRuntimeInstance
		cloneInstance.setUp(slot, it.executionContext)
		slot.adaptInstance(cloneInstance)
	}

	protected def dispatch cloneInstance(Object it, ExecutionSlot slot) {
	}

	protected def heap() {
		instanceContext.heap
	}
}
