/**
 * Copyright (c) 2022-2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Finlay Weegen - itemis AG
 * 	Axel terfloth - itemis AG
 * 
 */
package com.yakindu.sct.simulation.core.coverage

import com.yakindu.base.types.Property
import com.yakindu.sct.model.sgraph.Entry
import com.yakindu.sct.model.sgraph.Exit
import com.yakindu.sct.model.sgraph.Region
import com.yakindu.sct.model.sgraph.State
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.Synchronization
import com.yakindu.sct.model.sgraph.Transition
import com.yakindu.sct.model.sgraph.Vertex
import com.yakindu.sct.model.sgraph.util.SubchartDFS
import java.util.ArrayDeque
import java.util.Deque

class StatechartMeasurementBuilder {
	
	
	def dispatch Measurement buildMeasurement(Statechart statechart) {
		
		val builder = new DFSMeasurementBuilder(this) 
		builder.perform(statechart)
		return builder.result
	}

	/** 
	 * To avoid infinite recursion statechart measurements are created using a DFS.
	 */
	static class DFSMeasurementBuilder extends SubchartDFS {
		
		Measurement result
		Deque<Measurement> measurementStack = new ArrayDeque<Measurement>
		protected extension StatechartMeasurementBuilder builder
		
		new(StatechartMeasurementBuilder builder) {
			super()
			this.builder = builder
		}
		
		override beginVisit(Object element, Object parent, int depth) {
			
			var Measurement childMeasurement = new Measurement // use an empty measurement to avoid putting null onto stack
			
			if (element instanceof Statechart) {
				childMeasurement = element.build
			} else if (element instanceof Property) {
				val statechart = element.statechart
				val parentMeasurement = measurementStack.peek
				if (!parentMeasurement?.children.exists[subject === statechart]){
					childMeasurement = element.statechart.build
					parentMeasurement.children += childMeasurement
				}
			}
			
			measurementStack.push(childMeasurement)
		}
		
		override endVisit(Object element, Object parent, int depth, int minDepth) {
			result = measurementStack.pop
		}
	}
	
	def Measurement build(Statechart statechart) {
		new Measurement => [
			subject = statechart
			name = statechart.name
			type = Statechart
			timestamp = System.currentTimeMillis
			statechart.regions.forEach [ r |
				children += r.buildMeasurement
			]
		]
	}
	
	def dispatch Measurement buildMeasurement(Region region) {

		new Measurement => [
			subject = region
			type = Region
			region.vertices.forEach [ v |
				children += v.buildMeasurement
			]
		]
	}

	def dispatch Measurement buildMeasurement(State state) {

		_buildMeasurement(state as Vertex) => [
			type = State
			state.regions.forEach [ r |
				children += r.buildMeasurement
			]
		]
	}

	def dispatch Measurement buildMeasurement(Entry entry) {

		new Measurement => [
			subject = entry
			type = Entry
			entry.outgoingTransitions.forEach [ t |
				children += t.buildMeasurement
			]
		]
	}

	def dispatch Measurement buildMeasurement(Exit exit) {

		new Measurement => [
			subject = exit
			type = Exit
		]
	}

	def dispatch Measurement buildMeasurement(Vertex vertex) {

		new Measurement => [
			subject = vertex
			type = Vertex
			vertex.outgoingTransitions.forEach [ t |
				children += t.buildMeasurement
			]
		]
	}

	def dispatch Measurement buildMeasurement(Transition transition) {

		new Measurement => [
			subject = transition
			type = Transition
		]
	}

	def dispatch Measurement buildMeasurement(Synchronization sync) {

		new Measurement => [
			subject = sync
			type = Synchronization
			sync.outgoingTransitions.forEach [ t |
				children += t.buildMeasurement
			]
		]
	}
}

