/**
 * Copyright (c) 2022-2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * Finlay Weegen - itemis AG
 * 
 */
package com.yakindu.sctunit.coverage.report

import com.yakindu.base.types.Reaction
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.Vertex
import com.yakindu.sct.simulation.core.coverage.Measurement
import com.yakindu.sct.simulation.core.coverage.MeasurementExtension
import java.io.File
import java.io.FileOutputStream
import java.io.FileWriter
import java.nio.file.Files
import java.nio.file.Path
import java.nio.file.Paths
import java.util.ArrayDeque
import java.util.ArrayList
import java.util.Collection
import java.util.zip.ZipEntry
import java.util.zip.ZipOutputStream
import org.eclipse.core.runtime.FileLocator
import org.eclipse.core.runtime.NullProgressMonitor
import org.eclipse.core.runtime.Platform
import org.eclipse.emf.common.util.URI
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl
import org.eclipse.emf.ecore.util.EcoreUtil
import org.eclipse.emf.ecore.xmi.impl.XMIResourceImpl
import org.eclipse.emf.transaction.RecordingCommand
import org.eclipse.emf.transaction.TransactionalEditingDomain
import org.eclipse.gmf.runtime.diagram.core.preferences.PreferencesHint
import org.eclipse.gmf.runtime.diagram.ui.image.ImageFileFormat
import org.eclipse.gmf.runtime.diagram.ui.render.util.CopyToImageUtil
import org.eclipse.gmf.runtime.draw2d.ui.figures.FigureUtilities
import org.eclipse.gmf.runtime.notation.ConnectorStyle
import org.eclipse.gmf.runtime.notation.Diagram
import org.eclipse.gmf.runtime.notation.NotationPackage
import org.eclipse.gmf.runtime.notation.ShapeStyle
import org.eclipse.gmf.runtime.notation.View
import org.eclipse.swt.graphics.RGB
import static extension com.yakindu.sct.model.sgraph.util.SubchartDFS.*


class CoverageReportStore {

	var CopyToImageUtil renderer
	extension MeasurementExtension mExtension
	extension ReportExtensions rExtension

	var Measurement measurement
	var ArrayList<ReportElement> reportElements = new ArrayList

	new(Measurement measurement, CopyToImageUtil renderer) {
		this.measurement = measurement
		this.renderer = renderer
		this.mExtension = new MeasurementExtension
		this.rExtension = new ReportExtensions
	}

	/* saving the coverage report and necessary resources */
	def saveReportFolder(Path path, String filename) {
		var zipPath = Paths.get(path.parent.toString + "/" + filename).toString
		try ( 
			var outStream = new FileOutputStream(zipPath);
			var zipOutStream = new ZipOutputStream(outStream)
			
		) {

			populateReportElements(measurement, zipOutStream)
			var report = (new HTMLCoverageReport(measurement)).generate(reportElements) // generating the report based on the report elements 
			addCoverageFolderToZipStream(zipPath, zipOutStream)
			var reportFile = new File("index.html")
			reportFile.createNewFile
			try (var fileWriter = new FileWriter(reportFile)) {
				fileWriter.write(report)
				addFileWithNameToZipStream(report.bytes, "index.html", zipOutStream)
			} 
		} catch (Exception exception) {
			exception.printStackTrace
		} 
	}

	def void populateReportElements(Measurement measurement, ZipOutputStream zipOutStream) {
		reportElements += measurement.subject.reportElements(measurement)
		reportElements.forEach[
			for (var i = 0; i < images.size; i++) {
				addFileWithNameToZipStream(images.get(i),
					"/img/" + it.imageName(i), zipOutStream)
			}
		]
	}

	def protected dispatch Iterable<ReportElement> reportElements(Statechart it, Measurement measurement) {
		return new ArrayList() => [ it += new ReportElement(measurement, measurement.provideImages)] 
	}
	
	def protected dispatch Iterable<ReportElement> reportElements(Object it, Measurement measurement) {
		return measurement.children.map[ subject.reportElements(it) ].flatten
	}

	def protected dispatch Iterable<ReportElement> reportElements(Void it, Measurement measurement) {
		return measurement.children.map[ subject.reportElements(it) ].flatten
	}

// recursively adding the coverage resource folder and it's contents
	def void addCoverageFolderToZipStream(String zipPath, ZipOutputStream zipOutStream) {
		var bundle = Platform.getBundle("com.yakindu.sctunit.coverage.ui")
		val resources = new ArrayDeque<String>
		resources.offer("ressource/coverageReport/")

		while (! resources.empty) {
			val resource = resources.poll
			if (resource.endsWith("/")) {
				bundle.getEntryPaths(resource).asIterator.forEach[resources += it]
			} else {
				var path = new org.eclipse.core.runtime.Path(resource)
				var url = FileLocator.find(bundle, path, null);
				var inputStream = url.openStream
				val bytes = inputStream.readAllBytes
				addFileWithNameToZipStream(bytes, path.removeFirstSegments(2).toString, zipOutStream)
			}
		}
	}

// adding an individual file to the zipOutputStream
	def void readBytesFromFile(File file, ZipOutputStream zipOutStream, String zipPath) {
		val bytes = Files.readAllBytes(Paths.get(file.path))
		addFileWithNameToZipStream(bytes, file.name, zipOutStream)
	}

// adding the contents of the file together with the filename to the zipOutputStream
	def void addFileWithNameToZipStream(byte[] fileBytes, String filename, ZipOutputStream zipOutStream) {
		zipOutStream.putNextEntry(new ZipEntry(filename))
		zipOutStream.write(fileBytes)
	}

	def ArrayList<byte[]> provideImages(Measurement measurement) {
		var images = new ArrayList<byte[]>
		var mainStatechart = measurement.subject as Statechart
		val subcharts = mainStatechart.subcharts.map[p|p.statechart].toSet
		

		var diagrams = returnDiagramsFromStatechart(mainStatechart)

		for (diagram : diagrams) {

			val diagramCopy = EcoreUtil.copy(diagram)
			images.add(diagramCopy.generateImage(measurement))
		}

		if (subcharts != []) {
			for (subchart : subcharts) {
				for (diagram : returnDiagramsFromStatechart(subchart)) {
					val diagramCopy = EcoreUtil.copy(diagram)
					images.add(diagramCopy.generateImage(measurement))
				}
			}
		}
		images
	}

	def Collection<Diagram> returnDiagramsFromStatechart(Statechart statechart) {
		var diagramCollection = EcoreUtil.getObjectsByType(statechart.eResource().getContents(),
			NotationPackage.Literals.DIAGRAM)
		diagramCollection
	}
	
	

	// generates images based on diagram
	def byte[] generateImage(Diagram diagram, Measurement measurement) {
		
		diagram.eSetDeliver(false)

		var resource = new XMIResourceImpl(URI.createURI(new String("coverage" + measurement.name)))
		resource.contents += diagram

		var set = new ResourceSetImpl
		set.resources += resource

		val editingDomain = TransactionalEditingDomain.Factory.INSTANCE.createEditingDomain(set);

		editingDomain.commandStack.execute(new RecordingCommand(editingDomain) {
			override doExecute() {	
				highlightCoverage(diagram, measurement, new ArrayList())
			}
		})

		val image = renderer.copyToImageByteArray(diagram, -1, -1, ImageFileFormat.resolveImageFormat("svg"),
			new NullProgressMonitor(), PreferencesHint.USE_DEFAULTS, false)

		editingDomain.dispose

		return image
	}

	def protected void highlightCoverage(Diagram diagram, Measurement measurement, ArrayList<View> highlightingList) {
		diagram
			.eAllContents
			.filter(View)
			.filter[ it.element.considered ]
			.forEach[ 
				applyHighlightingColour(it, it.getHighlightingColour(measurement))
			]			
	}
    
	def protected applyHighlightingColour(View it, RGB colour) {
		if (it.element instanceof Reaction) {
			it.styles.filter(ConnectorStyle)?.head?.setLineColor(FigureUtilities.RGBToInteger(colour))
		} else if (it.element instanceof Vertex) {
			it.styles?.filter(ShapeStyle)?.head?.setFillColor(FigureUtilities.RGBToInteger(colour))
		}
	}

	def protected getHighlightingColour(View view, Measurement measurement) {
		val elem = view.element
		val stc = measurement.forSubject(elem).coverage
		val coverage = (stc === null) ? 0.0f : stc.coverage;
		if (coverage == 1.0f) {
			return CoverageColors.GREEN.RGB
		} else if (coverage == 0.0f) {
			return CoverageColors.RED.RGB
		} else
			return CoverageColors.YELLOW.RGB
	}

	def protected boolean considered(EObject eObject) {
		if (eObject instanceof Reaction || eObject instanceof Vertex) {
			return true;
		} else {
			return false;
		}
	}
}