/**
 * Copyright (c) 2020 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.base.expressions.linking

import com.google.common.collect.Iterables
import com.google.inject.Inject
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.util.ExpressionExtensions
import com.yakindu.base.types.Operation
import com.yakindu.base.types.TypesPackage
import java.util.Collections
import java.util.List
import java.util.Optional
import org.eclipse.emf.ecore.EClass
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.EReference
import org.eclipse.xtext.linking.impl.DefaultLinkingService
import org.eclipse.xtext.linking.impl.IllegalNodeException
import org.eclipse.xtext.naming.IQualifiedNameConverter
import org.eclipse.xtext.naming.QualifiedName
import org.eclipse.xtext.nodemodel.INode
import org.eclipse.xtext.resource.IEObjectDescription
import org.eclipse.xtext.scoping.IScope

/**
 * 
 * Supports operation overloading
 * 
 * @author andreas muelder - Initial contribution and API
 * 
 */
class ArgumentLinkingService extends DefaultLinkingService {
	@Inject
	protected extension IQualifiedNameConverter qualifiedNameConverter;
	@Inject
	protected extension OperationOverloadingResolver operationsLinker;

	@Inject protected extension ExpressionExtensions utils;

	override List<EObject> getLinkedObjects(EObject context, EReference ref, INode node) throws IllegalNodeException {
		if (context instanceof ArgumentExpression && isOperationCall(context)) {
			return getLinkedOperation((context as ArgumentExpression), ref, node);
		}
		return super.getLinkedObjects(context, ref, node);
	}

	def List<EObject> getLinkedOperation(ArgumentExpression context, EReference ref, INode node) {
		val EClass requiredType = ref.getEReferenceType();
		if (requiredType === null) {
			return Collections.<EObject>emptyList();
		}
		val crossRefString = getCrossRefNodeAsString(node);
		if (crossRefString === null || crossRefString.equals("")) {
			return Collections.<EObject>emptyList();
		}
		val IScope scope = getScope(context, ref);
		val QualifiedName qualifiedLinkName = qualifiedNameConverter.toQualifiedName(crossRefString);
		// Adoption to super class implementation here to return multi elements
		val Iterable<IEObjectDescription> eObjectDescription = scope.getElements(qualifiedLinkName);
		var size = Iterables.size(eObjectDescription);
		if (size == 0)
			return Collections.emptyList();
		if (size == 1)
			return Collections.singletonList(Iterables.getFirst(eObjectDescription, null).getEObjectOrProxy());
		// Two operation with same name found here
		var List<IEObjectDescription> candidates = newArrayList
		for (IEObjectDescription currentDescription : eObjectDescription) {
			if (TypesPackage.Literals.OPERATION.isSuperTypeOf(currentDescription.EClass)) {
				candidates.add(currentDescription);
			}
		}
		var Optional<Operation> operation = operationsLinker.linkOperation(candidates, context);
		if (operation.isPresent()) {
			return Collections.singletonList(operation.get());
		}
		// Link to first operation to get parameter errors instead of linking errors
		return Collections.singletonList(Iterables.getFirst(eObjectDescription, null).getEObjectOrProxy());
	}

	def protected boolean isOperationCall(EObject context) {
		if (context instanceof ElementReferenceExpression) {
			return (context as ElementReferenceExpression).isOperationCall();
		}
		if (context instanceof FeatureCall) {
			return (context as FeatureCall).isOperationCall();
		}
		return false;
	}
}
