/**
 * Copyright (c) 2020-2023 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.base.types;

import com.google.inject.Inject
import com.google.inject.Singleton
import com.yakindu.base.base.BasePackage
import com.yakindu.base.base.NamedElement
import java.util.Iterator
import java.util.List
import java.util.Set
import java.util.SortedSet
import org.eclipse.emf.common.util.BasicEList
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.EStructuralFeature
import org.eclipse.xtext.naming.QualifiedName

import static extension org.eclipse.xtext.EcoreUtil2.*

/**
 * @author Thomas Kutz - Initial contribution and API
 */
@Singleton
class TypesUtil {
	
	@Inject protected extension TypeBuilder

	public static final String ID_SEPARATOR = ".";
	public static final String SOURCE_TYPE = "__source_type__";
	
	def <T extends AnnotatableElement> T _source_type(T e) {
		e._annotate(SOURCE_TYPE)
	}
	
	def isSource_type(AnnotatableElement it){
		if(getAnnotationOfType(SOURCE_TYPE) !== null) return true
		else false
	}

	def static String computeQID(NamedElement element) {
		if (element.getName() === null) {
			return null;
		}

		val StringBuilder id = new StringBuilder();
		id.append(element.getName());
		var EObject container = element.eContainer();
		while (container !== null) {

			if (container.eClass().getEAllStructuralFeatures().contains(BasePackage.Literals.NAMED_ELEMENT__NAME)) {
				prependNamedElementName(id, container);
			} else {
				prependContainingFeatureName(id, container);
			}
			container = container.eContainer();
		}
		return id.toString();
	}
	
	def private static void prependNamedElementName(StringBuilder id, EObject container) {
		val String name = container.eGet(BasePackage.Literals.NAMED_ELEMENT__NAME) as String;
		if (name !== null) {
			id.insert(0, ID_SEPARATOR);
			id.insert(0, name);
		}
	}

	def private static void prependContainingFeatureName(StringBuilder id, EObject container) {
		val EStructuralFeature feature = container.eContainingFeature();
		if (feature !== null) {
			var String name;
			if (feature.isMany()) {
				val Object elements = container.eContainer().eGet(feature);
				var int index = 0;
				if (elements instanceof BasicEList) {
					val BasicEList<?> elementList = elements;
					index = elementList.indexOf(container);
				}
				name = feature.getName() + index;
			} else {
				name = feature.getName();
			}
			id.insert(0, ID_SEPARATOR);
			id.insert(0, name);
		}
	}
	
	def static dispatch toQualifiedName(TypeParameter tp) {
		return QualifiedName.create(tp.name)
	}
	
	def static dispatch toQualifiedName(Type t) {
		val names = t.allContainers.filter(Type).filter(p | !p.name.nullOrEmpty).map[name].toList.reverse
		names += t.name
		return QualifiedName.create(names)
	}
	
	def static dispatch toQualifiedName(ComplexType t) {
		val names = t.allContainers.filter(Package).filter(p | !p.name.nullOrEmpty).map[name].toList.reverse
		//
		names += t.name //.replace("[^A-Za-z0-9]","") //Remove special characters from type names
		return QualifiedName.create(names)
	}
	
	def static dispatch toQualifiedName(EObject t) {
		throw new IllegalArgumentException("Could not create qualified name for " + t)
	}
	
	def static Set<Type> collectPackageContainedTypes(Package pckg, Set<Type> types) {
		if(pckg !== null)
			pckg.member.forEach[
				if(it instanceof Package) it.collectPackageContainedTypes(types)
				else if(it instanceof Type) types += it
			]
		types
	}
	
	
	//TODO: Remove once this method is provided by Xbase library (Xtext version 2.35)
	def <T> T lastOrNull(Iterable<T> iterable) {
		if (iterable instanceof List<?>) {
			var list = iterable as List<T>;
			if (list.isEmpty())
				return null;
			return list.get(list.size() - 1);
		} else if (iterable instanceof SortedSet) {
			var sortedSet = iterable as SortedSet<T>;
			if (sortedSet.isEmpty())
				return null;
			return sortedSet.last();
		} else {
			return lastOrNull(iterable.iterator());
		}
	}
	
	def <T> T lastOrNull(Iterator<T> iterator) {
		var result = null as T
		while(iterator.hasNext()) {
			result = iterator.next();
		}
		return result;
	}
	

}
