/**
 * Copyright (c) 2022-2023 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.generator.c.files

import com.google.inject.Inject
import com.yakindu.sct.generator.c.extensions.FileNaming
import com.yakindu.sct.generator.c.extensions.GenmodelEntries
import com.yakindu.sct.generator.c.types.CTypes
import com.yakindu.sct.generator.core.artifacts.IContentTemplate
import com.yakindu.sct.generator.core.artifacts.IGenArtifactConfigurations
import com.yakindu.sct.model.sexec.ExecutionFlow
import com.yakindu.sct.model.sgen.GeneratorEntry

/**
 * @author Finlay Weegen
 */
class TimerServiceSource implements IContentTemplate<ExecutionFlow> {

	@Inject extension GenmodelEntries
	@Inject protected extension FileNaming
	@Inject extension CTypes

	override content(ExecutionFlow flow, GeneratorEntry entry, IGenArtifactConfigurations locations) '''
		«entry.licenseText»
		
		#include "«timerServiceModule.h»"
		#include <stddef.h>
		
		void reset_task_data(task_data *data)
		{
		    switch (data->type) {
		    case TIME_EVENT_TASK:
		        data->get.time_ms = 0;
		        data->get.time_event.raise_event = sc_null;
		        data->get.time_event.pt_evid = 0;
		        data->get.time_event.periodic = bool_false;
		        break;
		    case RUNCYCLE_TASK:
		        data->get.time_ms = 0;
		        data->get.run_cycle = sc_null;
		        break;
		    default:
		        return;
		    }
		    data->type = EMPTY_TASK;
		    data->get.statemachine = NULL;
		}
		
		void execute(task_data *data)
		{
		    switch (data->type) {
		    case TIME_EVENT_TASK: {
		    	/* Fire the event. */
		        sc_raise_time_event_fp raise_event = data->get.time_event.raise_event;
		        void *statemachine = data->get.statemachine;
		        if (raise_event != NULL && statemachine != NULL) {
		            raise_event(statemachine, data->get.time_event.pt_evid);
		        }
		        return;
		    }
		    case RUNCYCLE_TASK: {
		        sc_run_cycle_fp run_cycle = data->get.run_cycle;
		        void *statemachine = data->get.statemachine;
		        if (run_cycle != NULL) {
		            run_cycle(statemachine);
		        }
		        return;
		    }
		    default:
		        return;
		    }
		}
		
		void update_elapsed_time_ms(sc_timer_t *task, «sc_time.name» elapsed_time_ms_)
		{
		    task->elapsed_time_ms += elapsed_time_ms_;
		}
		
		«sc_bool.name» is_periodic(sc_timer_t *task)
		{
		    task_type type;
		    type = task->data.type;
		    switch (type) {
		    case TIME_EVENT_TASK:
		        return task->data.get.time_event.periodic;
		    case RUNCYCLE_TASK:
		        return bool_true;
		    default:
		        return bool_false;
		    }
		}

		«sc_bool.name» is_runcycle_event(sc_timer_t *task)
		{
		    return task->data.type == RUNCYCLE_TASK;
		}
		
		void reset_timer_task(sc_timer_t *task)
		{
		    reset_task_data(&task->data);
		    task->elapsed_time_ms = 0;
		}
		
		/*
		 Compare tasks based on their execution order.
		 Return true if this task is always to be executed strictly before the other task if both are scheduled to run at the same time.
		 Default behavior:
		 - This task is to be scheduled strictly before the other task if its is_runcycle_event() method does not return true and the other task's is_runcycle_event() method returns true.
		 */
		«sc_bool.name» less_than(sc_timer_t *task, sc_timer_t *other)
		{
		    return !is_runcycle_event(task) && is_runcycle_event(other);
		}
		
		«sc_bool.name» time_event_matcher(match_time_event *matcher, const sc_timer_t *other)
		{
		    return (matcher->statemachine == NULL || matcher->statemachine == other->data.get.statemachine) && matcher->pt_evid == other->data.get.time_event.pt_evid;
		}
		
		«sc_bool.name» run_cycle_matcher(match_run_cycle_of *matcher, const sc_timer_t *other)
		{
		    return matcher->statemachine == other->data.get.statemachine;
		}
		
		void set_generic_timer(sc_timer_service_t *timer_service, const task_data data)
		{
		    «sc_integer.name» inserted_task_idx;
		    sc_timer_t *inserted_task;
			
		    /* Do nothing if there are no free slots. */
		    if (timer_service->next_free_task >= timer_service->length) {
		        return;
		    }

		    /* Insert task at the front. */
		    inserted_task_idx = timer_service->next_free_task;
		    inserted_task = &(timer_service->tasks[inserted_task_idx]);
		    inserted_task->data = data;
		    inserted_task->elapsed_time_ms = 0;
		    timer_service->next_free_task = inserted_task->next_task_idx;
		    inserted_task->next_task_idx = timer_service->next_active_task;
		    timer_service->next_active_task = inserted_task_idx;
		}

		void unset_generic_timer(sc_timer_service_t *timer_service, timer_task_matcher *matcher)
		{
		    «sc_integer.name» last_position;
		    «sc_integer.name» next_position;
		    sc_timer_t *current_task;
		    «sc_bool.name» match_result;
		    «sc_integer.name» current_position;
		    
		    last_position = timer_service->length;
		    next_position = timer_service->next_active_task;
		
		    while (next_position < timer_service->length) {

		        current_task = &timer_service->tasks[next_position];
		        match_result = bool_false;
		
		        if (matcher->is_match_time_event) {
		            match_result = time_event_matcher(&matcher->data.match_time_event_, current_task);
		        } else {
		            match_result = run_cycle_matcher(&matcher->data.match_run_cycle_of_, current_task);
		        }
		
		        if (match_result) {
		            reset_timer_task(current_task);
		            if (last_position < timer_service->length) {
		                timer_service->tasks[last_position].next_task_idx = current_task->next_task_idx;
		            } else {
		                timer_service->next_active_task = current_task->next_task_idx;
		            }
		            
		            current_position = next_position;
		            next_position = current_task->next_task_idx;
		            current_task->next_task_idx = timer_service->next_free_task;
		            timer_service->next_free_task = current_position;
		        } else {
		            last_position = next_position;
		            next_position = current_task->next_task_idx;
		        }
		    }
		}
		
		void sc_timer_service_init(sc_timer_service_t *timer_service,
		                           sc_timer_t *tasks_,
		                           «sc_integer.name» length_,
		                           sc_raise_time_event_fp raise_event)
		{
		    «sc_integer.name» i;
			
		    timer_service->length = length_;
		    timer_service->tasks = tasks_;
		    timer_service->next_active_task = length_;
		    timer_service->next_free_task = 0;

		    for (i = 0; i < timer_service->length; i++) {
		        timer_service->tasks[i].next_task_idx = i + 1;
		        timer_service->tasks[i].elapsed_time_ms = 0;
		        timer_service->tasks[i].data.type = EMPTY_TASK;
		    }
		
		    timer_service->raise_event = raise_event;
		}
		
		/*! Start the timing for a time event.*/
		void sc_timer_set(sc_timer_service_t *timer_service,
		                  void *handle,
		                  const «sc_eventid.name» evid,
		                  const «sc_time.name» time_ms,
		                  const «sc_bool.name» periodic)
		{
		    sc_raise_time_event_fp raise_event = timer_service->raise_event;
		
		    sc_timer_set_for_raise_event(timer_service, raise_event, handle, evid, time_ms, periodic);
		}
		
		
		void sc_timer_set_for_raise_event(sc_timer_service_t *timer_service,
		                              sc_raise_time_event_fp raise_event,
		                              void *statemachine,
		                              const «sc_eventid.name» evid,
		                              const «sc_time.name» time_ms,
		                              const «sc_bool.name» periodic)
		{
		    task_data data;
		    data.type = TIME_EVENT_TASK;
		    data.get.time_ms = time_ms;
		    data.get.statemachine = statemachine;
		    data.get.run_cycle = NULL;
		    data.get.time_event.raise_event = raise_event;
		    data.get.time_event.pt_evid = evid;
		    data.get.time_event.periodic = periodic;
		
		    set_generic_timer(timer_service, data);
		}

		/*! Unset the given time event.*/
		void sc_timer_unset(sc_timer_service_t *timer_service, «sc_eventid.name» evid)
		{
		    void *statemachine;
		    statemachine = NULL;
		    sc_timer_unset_for_machine(timer_service, evid, statemachine);
		}
		
		void sc_timer_unset_for_machine(sc_timer_service_t *timer_service,
		                                 «sc_eventid.name» evid,
		                                 void *statemachine)
		{
		    timer_task_matcher matcher;
		    matcher.is_match_time_event = bool_true;
		    matcher.data.match_time_event_.statemachine = statemachine;
		    matcher.data.match_time_event_.pt_evid = evid;
		
		    unset_generic_timer(timer_service, &matcher);
		}
		
		/*! Set a timer for running cycles of the given statemachine.*/
		void set_runcycle_timer_for(sc_timer_service_t *timer_service,
		                             sc_run_cycle_fp run_cycle,
		                             void *statemachine,
		                             «sc_time.name» cycle_period)
		{
		    task_data data;
		    data.type = RUNCYCLE_TASK;
		    data.get.time_event.raise_event = NULL;
		    data.get.time_event.periodic = bool_false;
		    data.get.time_event.pt_evid = 0;
		    data.get.time_ms = cycle_period;
		    data.get.run_cycle = run_cycle;
		    data.get.statemachine = statemachine;
		
		    set_generic_timer(timer_service, data);
		}

		/*! Unset timers for running cycles of the given statemachine.*/
		void unset_runcycle_timer(sc_timer_service_t *timer_service, void *statemachine)
		{
		    timer_task_matcher matcher;
		    matcher.is_match_time_event = bool_false;
		    matcher.data.match_run_cycle_of_.statemachine = statemachine;
		
		    unset_generic_timer(timer_service, &matcher);
		}
		
		/*!
		 * This function must be called by the user. The elapsed time must be calculated every time,
		 * the function gets called.
		 */
		void sc_timer_service_proceed(sc_timer_service_t *timer_service, «sc_time.name» elapsed_ms)
		{
		    «sc_time.name» time_to_proceed;
		    «sc_time.name» proceeded_time;
		    «sc_integer.name» idx;
		    «sc_bool.name» task_fired;
		    «sc_integer.name» before_best;
		    «sc_integer.name» best;
		    «sc_time.name» best_remaining_time;
		    «sc_integer.name» last_task;
		    «sc_integer.name» next_task;
		    «sc_time.name» remaining_time;
		    «sc_time.name» time_till_next_task_;
		    
		    if (timer_service->next_active_task >= timer_service->length) {
		        return;
		    }
		    
		    time_till_next_task_ = time_till_next_task(timer_service);
		    
		    time_to_proceed = time_till_next_task_ < elapsed_ms
		                                     ? time_till_next_task_
		                                     : elapsed_ms;
		    proceeded_time = 0;
		
		    while (time_to_proceed > 0) {
		        
		        idx = timer_service->next_active_task;
		
		        while (idx < timer_service->length) {
		            sc_timer_t *current_task;
		            current_task = &timer_service->tasks[idx];
		            update_elapsed_time_ms(current_task, time_to_proceed);
		            idx = current_task->next_task_idx;
		        }
		
		        do {
		            task_fired = bool_false;
		            
		            
		            before_best = timer_service->length;
		            best = timer_service->next_active_task;
		            best_remaining_time = timer_service->tasks[best].data.get.time_ms
		                                             - timer_service->tasks[best].elapsed_time_ms;
		            last_task = best;
		            next_task = timer_service->tasks[best].next_task_idx;
		            while (next_task < timer_service->length) {
		                sc_timer_t current_task;
		                
		                
		                current_task = timer_service->tasks[next_task];
		                remaining_time = current_task.data.get.time_ms
		                                            - current_task.elapsed_time_ms;
		                if (remaining_time < best_remaining_time
		                    || (remaining_time == best_remaining_time
		                        && !(less_than(&timer_service->tasks[best], &current_task)))) {
		                    best = next_task;
		                    before_best = last_task;
		                    best_remaining_time = remaining_time;
		                }
		                last_task = next_task;
		                next_task = current_task.next_task_idx;
		            }
		            if (best_remaining_time <= 0) {
		                sc_timer_t *best_task;
		                task_data data;
		                
		                best_task = &timer_service->tasks[best];
		                data = best_task->data;
		                if (is_periodic(best_task)) {
		                    best_task->elapsed_time_ms = 0;
		                } else {
		                    reset_timer_task(best_task);
		                    if (before_best < timer_service->length) {
		                        timer_service->tasks[before_best].next_task_idx = best_task->next_task_idx;
		                    } else {
		                        timer_service->next_active_task = best_task->next_task_idx;
		                    }
		                    best_task->next_task_idx = timer_service->next_free_task;
		                    timer_service->next_free_task = best;
		                }
		                execute(&data);
		                task_fired = bool_true;
		            }
		        } while (task_fired && timer_service->next_active_task < timer_service->length);
		        proceeded_time += time_to_proceed;
		        time_till_next_task_  = time_till_next_task(timer_service);
		        time_to_proceed = (time_till_next_task_ < elapsed_ms - proceeded_time)
		                              ? time_till_next_task_
		                              : (elapsed_ms - proceeded_time);
		    }
		}
		
		/*! Cancel timer service. Use this to end possible timing threads and free
		 memory resources.
		 */
		void cancel(sc_timer_service_t *timer_service)
		{
		    «sc_integer.name» idx;
		    for (idx = 0; idx < timer_service->length; idx++) {
		        reset_timer_task(&timer_service->tasks[idx]);
		        timer_service->tasks[idx].next_task_idx = idx + 1;
		    }
		    timer_service->next_active_task = timer_service->length;
		    timer_service->next_free_task = 0;
		}
		
		/*! Obtain the time (in ms) required to proceed to the next task.
		 */		
		«sc_time.name» time_till_next_task(sc_timer_service_t *timer_service)
		{
		    «sc_time.name» time;
		    «sc_integer.name» task;
		    «sc_time.name» remaining_time;
					    
		    if (timer_service->next_active_task == timer_service->length) {
		        return 0;
		    }
		
		    time = timer_service->tasks[timer_service->next_active_task].data.get.time_ms
		                      - timer_service->tasks[timer_service->next_active_task].elapsed_time_ms;
		    task = timer_service->tasks[timer_service->next_active_task].next_task_idx;
		    while (task < timer_service->length) {
		        sc_timer_t *current_task;
		        
		        current_task = &timer_service->tasks[task];
		        remaining_time = current_task->data.get.time_ms - current_task->elapsed_time_ms;
		        
		        if (remaining_time < time) {
		            time = remaining_time;
		        }
		        task = current_task->next_task_idx;
		    }
		    return time;
		}
		
		
	'''

}
