/**

 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Andreas Muelder - itemis AG
 * 
 */
package com.yakindu.sct.generator.c.typesystem;

import com.google.inject.Singleton
import com.yakindu.base.types.Type
import com.yakindu.base.types.TypesFactory
import com.yakindu.base.types.typesystem.GenericTypeSystem

import static com.yakindu.base.types.TypesFactory.*

/**
 * @author andreas muelder - Initial contribution and API
 * 
 */
@Singleton
 class CTypeSystem extends GenericTypeSystem {

	static val CTypeSystem INSTANCE = new CTypeSystem()

	public static val String DOUBLE = "double"
	public static val String UINT8_T = "uint8_t"
	public static val String INT8_T = "int8_t"
	public static val String INT16_T = "int16_t"
	public static val String UINT16_T = "uint16_t"
	public static val String INT32_T = "int32_t"
	public static val String INT64_T = "int64_t"
	public static val String UINT32_T = "uint32_t"
	public static val String UINT64_T = "uint64_t"
	public static val String FLOAT = "float"
	public static val String BOOL = "bool"
	public static val String UNSUPPORTED_TYPE = "UnsupportedType"

	public static val String POINTER = "pointer"
	public static val String SHARED_POINTER = "shared_ptr"
	public static val String WEAK_POINTER = "weak_ptr"
	public static val String UNIQUE_POINTER = "unique_ptr"
	public static val SMART_POINTER_TYPES = #[SHARED_POINTER,WEAK_POINTER,UNIQUE_POINTER]
	public static val POINTER_PROPERTY = #[POINTER,SHARED_POINTER,WEAK_POINTER,UNIQUE_POINTER]
	public static val POINTER_TYPES = #[POINTER,SHARED_POINTER,WEAK_POINTER,UNIQUE_POINTER]
	
	public static val String POINTER_VALUE_PROPERTY = "value"

	protected new() {
	}

	static def CTypeSystem getInstance() {
		return INSTANCE
	}

	override initRegistries() {
		super.initRegistries()

		getType(BOOLEAN).abstract = true
		getType(INTEGER).abstract = true
		//getType(REAL).abstract = true
		declareRangedInteger(INT8_T, -128L, 127L);
		declareRangedInteger(INT16_T, -32768L, 32767L);
		declareRangedInteger(INT32_T, -2147483648L, 2147483647L)
		declareRangedInteger(INT64_T, 0, 0); // TODO: Size of int64
		declareRangedInteger(UINT8_T, 0L, 255L)
		declareRangedInteger(UINT16_T, 0L, 65535L)
		declareRangedInteger(UINT32_T, 0L, 4294967295L)
		declareRangedInteger(UINT64_T, 0, 0); // TODO: Size of int64
		declareSuperType(getType(INT64_T), getType(INTEGER))
		declareSuperType(getType(INT32_T), getType(INT64_T))
		declareSuperType(getType(INT16_T), getType(INT32_T))
		declareSuperType(getType(INT8_T), getType(INT16_T))

		declareSuperType(getType(UINT64_T), getType(UNSIGNED_INTEGER))
		declareSuperType(getType(UINT32_T), getType(UINT64_T))
		declareSuperType(getType(UINT16_T), getType(UINT32_T))
		declareSuperType(getType(UINT8_T), getType(UINT16_T))

		declarePrimitive(FLOAT)
		declarePrimitive(DOUBLE)
		declarePrimitive(BOOL)
		declarePrimitive(UNSUPPORTED_TYPE)
		getType(UNSUPPORTED_TYPE).abstract = true

		declareSuperType(getType(DOUBLE), getType(REAL))
		declareSuperType(getType(FLOAT), getType(DOUBLE))
		declareSuperType(getType(BOOL), getType(BOOLEAN))

		// make integers assignable to floats and doubles
		declareSuperType(getType(INTEGER), getType(FLOAT))

		// 'ANY' type is used to transform '#defines' into constants. As we
		// cannot always derive a proper type for them and they should be
		// assignable in expressions, 'ANY' type is used as leaf in the type
		// hierarchy tree
		declareSuperType(getType(ANY), getType(UINT8_T))
		declareSuperType(getType(ANY), getType(INT8_T))
		declareSuperType(getType(ANY), getType(BOOL))
		declareSuperType(getType(ANY), getType(STRING))

		declarePointerType(POINTER)
		declarePointerType(SHARED_POINTER)
		declarePointerType(WEAK_POINTER)
		declarePointerType(UNIQUE_POINTER)
		declareSuperType(getType(ARRAY), getType(POINTER))
		declareSuperType(getType(ARRAY), getType(SHARED_POINTER))
		declareSuperType(getType(ARRAY), getType(WEAK_POINTER))
		declareSuperType(getType(ARRAY), getType(UNIQUE_POINTER))
	}

	protected def declarePointerType(String pointerType) {
		val pointer = createPointerType(pointerType)
		declareType(pointer, pointer.getName())
		getResource().contents += pointer

		// Declare extension method to get a pointer for any type
		var pointerExtension = eINSTANCE.createProperty => [
			name = pointerType
			typeSpecifier = eINSTANCE.createArrayTypeSpecifier => [
				type = pointer
			]
		]
		getResource.contents += pointerExtension
		extensionPropertyRegistry.put(getType(ANY), pointerExtension)
	}

	protected def createPointerType(String pointerType) {
		eINSTANCE.createComplexType => [ type |
			type.name = pointerType
			type.annotations += TypesFactory.eINSTANCE.createAnnotation => [
				type = typeAnnotations.createBuiltInTypeAnnotationType()
			]
			type.typeParameters += eINSTANCE.createTypeParameter => [
				name = "baseType"
			]
			type.features += eINSTANCE.createProperty => [
				name = POINTER_VALUE_PROPERTY
				typeSpecifier = eINSTANCE.createTypeSpecifier => [
					type = type.typeParameters.head
				]
			]
		]
	}
	
	override getOperationExtensions(Type type) {
		val result = super.getOperationExtensions(type)
		result += super.getOperationExtensions(getType(ANY))
		result
	}

	override getPropertyExtensions(Type type) {
		val result = super.getPropertyExtensions(type)
		result += super.getPropertyExtensions(getType(ANY))
		result
	}

	/**
	 * lower and upper bounds are currently not used
	 */
	def declareRangedInteger(String name, long lowerBound, long upperBound) {
		val type = declarePrimitive(name)
		type
	}
}
