/**
 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Jonathan Thoene - itemis AG
 * 
 */

package com.yakindu.sctunit.simulation.core.interpreter

import com.google.inject.Inject
import com.yakindu.base.base.NamedElement
import com.yakindu.base.expressions.expressions.ArgumentExpression
import com.yakindu.base.expressions.expressions.ElementReferenceExpression
import com.yakindu.base.expressions.expressions.FeatureCall
import com.yakindu.base.expressions.interpreter.IOperationExecutor
import com.yakindu.base.expressions.interpreter.context.IExecutionSlotResolver
import com.yakindu.base.types.EnumerationType
import com.yakindu.base.types.Enumerator
import com.yakindu.base.types.Expression
import com.yakindu.base.types.Operation
import com.yakindu.base.types.Type
import com.yakindu.base.types.inferrer.ITypeSystemInferrer
import com.yakindu.sct.model.sgraph.Scope
import com.yakindu.sct.model.sruntime.ExecutionContext
import com.yakindu.sct.model.stext.stext.ActiveStateReferenceExpression
import com.yakindu.sct.model.stext.stext.VariableDefinition
import com.yakindu.sct.simulation.core.sexec.interpreter.DefaultOperationMockup
import com.yakindu.sct.simulation.core.util.ExecutionContextExtensions
import com.yakindu.sctunit.sCTUnit.AssertionStatement
import com.yakindu.sctunit.sCTUnit.TestStatement
import com.yakindu.sctunit.sCTUnit.VerifyCalledStatement
import com.yakindu.sctunit.simulation.core.junit.Failure
import java.util.List
import java.util.Set
import org.apache.commons.lang.StringEscapeUtils
import org.eclipse.emf.ecore.EObject
import org.eclipse.xtext.nodemodel.util.NodeModelUtils

/**
 * 
 * @author jonathan thoene - Initial contribution and API
 * 
 */
class DefaultSCTUnitFailureTracer implements IFailureTracer {

	@Inject protected ITypeSystemInferrer tsi
	@Inject protected IExecutionSlotResolver resolver
	@Inject protected extension ExecutionContextExtensions
	@Inject protected Set<IOperationExecutor> operationMockups

	override getFailureStack(AssertionStatement assertion, ExecutionContext it) {
		var allRefs = assertion.eAllContents.filter [ elem |
			(elem instanceof ArgumentExpression && doEvaluate(elem as ArgumentExpression)) ||
			(elem instanceof ActiveStateReferenceExpression)
		]
		val failures = allRefs.map[elem|elem.failureMessage(it)].toList
		var failure = new Failure() => [
			message = '''
				Assertion failed: "«getAssertionMessage(assertion)»" (Line «NodeModelUtils.getNode(assertion).startLine»)
					«FOR failure : failures»
						«failure»
					«ENDFOR»
			'''
		]
		return failure;
	}

	override getFailureStack(VerifyCalledStatement it, List<Object> calledParameters) {
		val operation = reference.operation.name
		if ((negated && !mockup.wasNotCalled(operation, calledParameters)) ||
			(!negated && !mockup.wasCalledAtLeast(operation, calledParameters, times))) {
			var failure = new Failure() => [ f |
				f.message = '''
					Assertion failed: "«getAssertionMessage(it)»" (Line «NodeModelUtils.getNode(it).startLine»)
						«verifyCalledStatementMessage(calledParameters)»
				'''
			]
			return failure
		} else {
			return null
		}
	}
	
	def getMockup() {
		return operationMockups.filter(DefaultOperationMockup).head
	}

	def verifyCalledStatementMessage(VerifyCalledStatement it, List<Object> calledParameters) {
		var long expectedCount = switch(it) {
			case negated: {0}
			case times !== null: {times.value}
			default: {1}
		}
		var parameters = if(calledParameters !== null && !calledParameters.empty) {
			"\nwith parameters " + calledParameters.toString + ",\n"
		} else {
			",\n"
		}
		'''
		Expected operation «reference.operation.name» to be called «times(expectedCount)» «parameters»
		but it was called «times(mockup.getOperationCallCount(reference.operation.name, calledParameters))».'''
	}
	
	def CharSequence times(long times) {
		if(times === 1) '''1 time''' else '''«times» times'''
	}

	def protected dispatch CharSequence getAssertionMessage(AssertionStatement it) {
		StringEscapeUtils.escapeXml(
			if (errorMsg !== null) {
				errorMsg
			} else {
				NodeModelUtils.getTokenText(NodeModelUtils.getNode(it))
			}.replace("\r", "").replace("\n", "").trim
		)
	}

	def protected dispatch CharSequence getAssertionMessage(TestStatement it) {
		StringEscapeUtils.escapeXml(
			NodeModelUtils.getNode(it).text.replace("\r", "").replace("\n", "").trim
		)
	}

	protected dispatch def String failureMessage(EObject it, ExecutionContext context) {
		''''''
	}

	protected dispatch def String failureMessage(ElementReferenceExpression it, ExecutionContext context) {
		'''name: «it.reference.elemName», value: «value(context, it)»'''
	}

	protected dispatch def String failureMessage(FeatureCall it, ExecutionContext context) {
		'''name: «it.owner.elemName».«it.feature.elemName», value: «value(context, it)»'''
	}

	protected dispatch def String failureMessage(ActiveStateReferenceExpression it, ExecutionContext context) {
		'''Expected: «value.name», actual: [«FOR state : context.allActiveStates SEPARATOR ', '»«state.name»«ENDFOR»]'''
	}

	private dispatch def String elemName(NamedElement it) {
		name
	}

	private dispatch def String elemName(EObject it) {
		""
	}
	
	private dispatch def String elemName(ElementReferenceExpression it) {
		reference.elemName
	}

	def dispatch Operation getOperation(Expression it) {
	}

	def dispatch Operation getOperation(FeatureCall it) {
		return feature as Operation
	}

	def dispatch Operation getOperation(ElementReferenceExpression it) {
		return reference as Operation
	}
	
	def dispatch value(ExecutionContext context, Expression it) {
		resolver.resolve(context,it).orElse(null)?.value
	}
	
	def dispatch value(ExecutionContext context, ElementReferenceExpression it) {
		val value = resolver.resolve(context,it).orElse(null)?.value
		val enumType = enumerationType
		if(enumType !== null) {
			return (reference as VariableDefinition).typeSpecifier.type.name +  "." + (value as Enumerator).name
		}
		return value
	}
	
	def EnumerationType enumerationType(ElementReferenceExpression it) {
		val typeResult = tsi.infer(it)
		val type = typeResult?.type
		if(type instanceof EnumerationType) {
			return type
		}
		return null
	}
	
	def dispatch boolean doEvaluate(ElementReferenceExpression it) {
		switch(it.reference) {
			Scope: false
			Type: false
			default: true
		}
	}
	
	def dispatch boolean doEvaluate(FeatureCall it) {
		switch(it.feature) {
			Enumerator: false
			default: true
		}
	}
	
	def dispatch boolean doEvaluate(ArgumentExpression it) {
		false
	}
}
