/**
 * Copyright (c) 2020 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.generator.genmodel.ui.quickfix

import com.google.inject.Inject
import com.yakindu.sct.commons.EmfUriUtil
import com.yakindu.sct.generator.core.extensions.GeneratorExtensions
import com.yakindu.sct.generator.core.extensions.ILibraryDescriptor
import com.yakindu.sct.generator.core.extensions.LibraryExtensions
import com.yakindu.sct.generator.core.library.AbstractDefaultFeatureValueProvider
import com.yakindu.sct.generator.genmodel.ui.internal.GenmodelActivator
import com.yakindu.sct.generator.genmodel.validation.SGenJavaValidator
import com.yakindu.sct.model.sgen.FeatureConfiguration
import com.yakindu.sct.model.sgen.GeneratorEntry
import com.yakindu.sct.model.sgen.GeneratorModel
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.ui.editor.partitioning.DiagramPartitioningUtil
import java.util.Collections
import org.eclipse.core.runtime.NullProgressMonitor
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.transaction.TransactionalEditingDomain
import org.eclipse.xtext.EcoreUtil2
import org.eclipse.xtext.ui.editor.model.edit.IModificationContext
import org.eclipse.xtext.ui.editor.model.edit.ISemanticModification
import org.eclipse.xtext.ui.editor.quickfix.DefaultQuickfixProvider
import org.eclipse.xtext.ui.editor.quickfix.Fix
import org.eclipse.xtext.ui.editor.quickfix.IssueResolutionAcceptor
import org.eclipse.xtext.validation.Issue

/** 
 * @author andreas muelder - Initial contribution and API
 */
class SGenQuickfixProvider extends DefaultQuickfixProvider {
	public static final String CHANGE_DOMAIN_COMMAND = "Domain change command"

	@Fix(SGenJavaValidator::CODE_REQUIRED_FEATURE)
	def void AddRequiredFeature(Issue issue, IssueResolutionAcceptor acceptor) {
		acceptor.accept(issue, '''Add feature «issue.getData().get(0)»'''.toString,
			'''Adds the feature «issue.getData().get(0)»'''.toString, null, (
			[ EObject element, IModificationContext context |
				var FeatureConfiguration config = getDefaultFeatureConfiguration(issue, element)
				if (config !== null) {
					var GeneratorEntry entry = (element as GeneratorEntry)
					entry.getFeatures().add(config)
				}
			] as ISemanticModification))
	}

	@Fix(AbstractDefaultFeatureValueProvider::CODE_MISSING_PROJECT)
	def void changeTargetProject(Issue issue, IssueResolutionAcceptor acceptor) {
		val currentProject = EmfUriUtil.toFile(issue.uriToProblem).project.name
		val iSemanticModification = [ EObject element, IModificationContext context |
			context.xtextDocument.replace(issue.offset, issue.length, '"' + currentProject + '"')
		] as ISemanticModification
		acceptor.accept(issue, "Change targetProject to '" + currentProject + "'", "", null, iSemanticModification)
	}

	@Fix(SGenJavaValidator::CODE_REQUIRED_DOMAIN)
	def void changeToValidDomain(Issue issue, IssueResolutionAcceptor acceptor) {
		var String[] validDomains = issue.getData().get(0).split(",")
		for (String validDomain : validDomains) {
			addAcceptor(issue, acceptor, validDomain)
		}
	}

	def private void addAcceptor(Issue issue, IssueResolutionAcceptor acceptor, String validDomain) {
		acceptor.accept(issue, validDomain, null, null, ([ EObject element, IModificationContext context |
			if (element instanceof GeneratorEntry) {
				var EObject referencedStatechart = element.getElementRef()
				if (referencedStatechart instanceof Statechart) {
					var TransactionalEditingDomain sharedDomain = DiagramPartitioningUtil::getSharedDomain()
					var DomainChangeCommand refactoringCommand = new DomainChangeCommand(sharedDomain,
						CHANGE_DOMAIN_COMMAND, Collections::EMPTY_LIST, validDomain, referencedStatechart)
					try {
						refactoringCommand.execute(new NullProgressMonitor(), null)
					} catch (Exception ex) {
						ex.printStackTrace()
					}
				}
			}
		] as ISemanticModification))
	}

	def private FeatureConfiguration getDefaultFeatureConfiguration(Issue issue, EObject element) {
		val model = (EcoreUtil2::getRootContainer(element) as GeneratorModel)
		val generatorDescriptor = GeneratorExtensions.getGeneratorDescriptor(model.getGeneratorId())
		if (!generatorDescriptor.isPresent())
			return null

		val libraryDescriptor = LibraryExtensions::getLibraryDescriptor(generatorDescriptor.get().getLibraryID())
		val library = libraryDescriptor.featureTypeLibrary
		val valueProvider = getValueProvider(libraryDescriptor)
		for (type : library.types) {
			if (type.name.equals(issue.data.head)) {
				return valueProvider.createDefaultFeatureConfiguration(type, element)
			}
		}
		return null
	}
	
	def private getValueProvider(ILibraryDescriptor desc) {
		desc.createFeatureValueProvider(
			GenmodelActivator::getInstance().getInjector(GenmodelActivator::COM_YAKINDU_SCT_GENERATOR_GENMODEL_SGEN))
	}

}
