/**
 * Copyright (c) 2020 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.domain.c.runtime.resource.transform

import com.google.inject.Inject
import com.yakindu.base.types.Type
import com.yakindu.base.types.TypesPackage
import com.yakindu.base.types.TypesUtil
import com.yakindu.sct.domain.c.runtime.resource.CHeaderResource
import com.yakindu.sct.domain.c.runtime.resource.transform.c.CTypeNaming
import java.io.File
import org.eclipse.cdt.core.dom.ast.IASTNameOwner
import org.eclipse.core.resources.IFile
import org.eclipse.core.resources.ResourcesPlugin
import org.eclipse.emf.common.util.URI
import org.eclipse.emf.ecore.EClass
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.InternalEObject
import org.eclipse.emf.workspace.util.WorkspaceSynchronizer

/**
 * @author rbeckmann
 *
 */
class ProxyBuilder {
	
	@Inject protected extension CTypeNaming
	
	static val String CPP_SEPARATOR = "::"
	
	def Type createTypeProxy(CHeaderResource it, String pathToDeclaringHeader, IASTNameOwner declSpecifier) {
		val headerUri = new File(pathToDeclaringHeader).toURI
		val headerFiles = ResourcesPlugin.workspace.root.findFilesForLocationURI(headerUri)

		if (headerFiles.isNullOrEmpty || !headerFiles.head.exists) {
			return createProxy(TypesPackage.Literals.TYPE,
				createProxyURI(declSpecifier.name, URI::createFileURI(pathToDeclaringHeader)).appendQuery(contextProjectName)) as Type
		} else {
			return createProxy(TypesPackage.Literals.TYPE, 
				createProxyURI(declSpecifier.name.toString, headerFiles.head)) as Type
		}
	}
	
	def createComplexTypeProxy(CHeaderResource it, String pathToDeclaringHeader, IASTNameOwner declSpecifier) {
		val headerUri = new File(pathToDeclaringHeader).toURI
		val headerFiles = ResourcesPlugin.workspace.root.findFilesForLocationURI(headerUri)

		if (headerFiles.isNullOrEmpty || !headerFiles.head.exists) {
			return createProxy(TypesPackage.Literals.COMPLEX_TYPE,
				createProxyURI(declSpecifier.name, URI::createFileURI(pathToDeclaringHeader)).appendQuery(contextProjectName)) as Type
		} else {
			return createProxy(TypesPackage.Literals.COMPLEX_TYPE, 
				createProxyURI(declSpecifier.name.toString, headerFiles.head)) as Type
		}
	}
	
	
	def protected createProxyURI(CHeaderResource it, String targetName, IFile headerFile) {
		// assumption: all types are top-level in header resource 
		val typeId = CHeaderResource.computePackageName(headerFile) + TypesUtil.ID_SEPARATOR + targetName.replaceAll(CPP_SEPARATOR, TypesUtil.ID_SEPARATOR)
		val typeUri = URI::createPlatformResourceURI(headerFile.fullPath.toOSString, true).appendFragment(typeId)
		return typeUri
	}

	def protected createProxyURI(CHeaderResource it, String targetName, URI fileURI) {
		// assumption: all types are top-level in header resource 
		val typeId = computePackageName(fileURI) + TypesUtil.ID_SEPARATOR + targetName.replaceAll(CPP_SEPARATOR, TypesUtil.ID_SEPARATOR)
		val typeUri = fileURI.appendFragment(typeId)
		return typeUri
	}

	def protected EObject createProxy(EClass eClass, URI uri) {
		val proxy = eClass.getEPackage().getEFactoryInstance().create(eClass) as InternalEObject;
		proxy.eSetProxyURI(uri);
		return proxy;
	}
	
	def protected getContextProjectName(CHeaderResource it) {
		getURI()?.query ?: WorkspaceSynchronizer.getFile(it)?.getProject()?.getName();
	}
}