/**
 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Andreas Muelder - itemis AG
 * 
 */
package com.yakindu.sct.generator.scxml

import com.google.inject.Inject
import com.yakindu.base.expressions.expressions.ReactionTrigger
import com.yakindu.base.expressions.expressions.TimeEventSpec
import com.yakindu.base.types.Declaration
import com.yakindu.base.types.Event
import com.yakindu.base.types.Property
import com.yakindu.sct.model.sgraph.Choice
import com.yakindu.sct.model.sgraph.CompositeElement
import com.yakindu.sct.model.sgraph.Entry
import com.yakindu.sct.model.sgraph.Exit
import com.yakindu.sct.model.sgraph.FinalState
import com.yakindu.sct.model.sgraph.Region
import com.yakindu.sct.model.sgraph.State
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.Synchronization
import com.yakindu.sct.model.sgraph.Transition
import com.yakindu.sct.model.sgraph.Vertex
import org.eclipse.emf.ecore.EObject
import org.eclipse.xtext.EcoreUtil2
import org.eclipse.xtext.naming.IQualifiedNameProvider
import org.eclipse.xtext.naming.IQualifiedNameProvider.AbstractImpl
import org.eclipse.xtext.naming.QualifiedName
import org.eclipse.xtext.naming.SimpleNameProvider

/**
 * 
 * @author andreas muelder - Initial contribution and API
 * 
 */
class SCXMLNameProvider extends AbstractImpl implements IQualifiedNameProvider {

	static final String _CHOICE_NAME = "choice_";
	static final String _ENTRY_NAME = "entry_";
	static final String _EXIT_NAME = "exit_";
	public static final String _FINAL_STATE_NAME = "final_";
	static final String _SYNC_NAME = "sync";
	static final String _TIME_EVENT = "_timeEvent_";

	@Inject
	protected SimpleNameProvider delegate

	def dispatch QualifiedName getFullyQualifiedName(Event it) {
		scopeQualifiedName(it)
	}

	def dispatch QualifiedName getFullyQualifiedName(Property it) {
		scopeQualifiedName(it)
	}

	protected def QualifiedName scopeQualifiedName(Declaration it) {
		val scope = delegate.getFullyQualifiedName(it.eContainer)
		if (scope !== null)
			return scope.append(delegate.getFullyQualifiedName(it))
		delegate.getFullyQualifiedName(it)
	}

	def dispatch QualifiedName getFullyQualifiedName(EObject it) {
		val parentFqn = delegate.getFullyQualifiedName(it)
		if (parentFqn === null)
			return null;
		QualifiedName.create(parentFqn.toString.trim.replaceAll("\\s+", "_"))
	}

	def dispatch QualifiedName getFullyQualifiedName(FinalState state) {
		uniqueForRegion(state, typeof(FinalState), _FINAL_STATE_NAME)
	}

	def dispatch QualifiedName getFullyQualifiedName(Synchronization it) {
		uniqueForRegion(typeof(Synchronization), _SYNC_NAME)
	}

	def dispatch QualifiedName getFullyQualifiedName(Choice it) {
		uniqueForRegion(typeof(Choice), _CHOICE_NAME)
	}

	def dispatch QualifiedName getFullyQualifiedName(Exit it) {
		val result = delegate.getFullyQualifiedName(it)
		return if (result === null) {
			uniqueForRegion(typeof(Exit), _EXIT_NAME)
		} else
			result
	}

	def dispatch QualifiedName getFullyQualifiedName(Entry it) {
		val result = delegate.getFullyQualifiedName(it)
		return if (result === null) {
			uniqueForRegion(typeof(Entry), _ENTRY_NAME)
		} else
			result
	}

	def dispatch QualifiedName getFullyQualifiedName(TimeEventSpec it) {
		val transition = EcoreUtil2.getContainerOfType(it, Transition)
		val state = EcoreUtil2.getContainerOfType(it, State)
		val name = state.name + if(transition !== null) "_t_" + state.outgoingTransitions.indexOf(transition) else ""
		val index = if(transition !== null) transition.eAllContents.filter(TimeEventSpec).toList.indexOf(it) else state.
				localReactions.map[lr|lr.eAllContents.toList.filter(TimeEventSpec)].flatten.toList.indexOf(it)
		QualifiedName.create(name + _TIME_EVENT + index);
	}

	def dispatch QualifiedName getFullyQualifiedName(Region ele) {
		if (ele.name.isNullOrEmpty) {
			if (ele.eContainer() instanceof CompositeElement) {
				val parent = ele.eContainer() as CompositeElement;
				val index = parent.getRegions().indexOf(ele);
				if (index != -1) {
					return QualifiedName.create("_r" + index);
				}
			}
		}
		return QualifiedName.create(ele.name.trim.replaceAll("\\s+", "_"))
	}

	def uniqueForRegion(Vertex v, Class<?> type, String prefix) {
		val composite = v.parentRegion.composite
		val index = v.parentRegion.vertices.filter(type).toList.indexOf(v)
		var fqn = if(composite instanceof State) composite.fullyQualifiedName else QualifiedName.EMPTY
		if (composite.isOrthogonal) {
			fqn = fqn.append(v.parentRegion.fullyQualifiedName)
		}
		return fqn.append(prefix + index)
	}

}
