/**
 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	Andreas Muelder - itemis AG
 * 
 */
package com.yakindu.sct.domain.scxml.simulation

import com.google.inject.Inject
import com.yakindu.base.expressions.interpreter.IExpressionInterpreter
import com.yakindu.base.expressions.interpreter.context.IExecutionContextInitializer
import com.yakindu.sct.model.sexec.Execution
import com.yakindu.sct.model.sexec.transformation.IModelSequencer
import com.yakindu.sct.model.sgraph.RegularState
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sgraph.Vertex
import com.yakindu.sct.model.sruntime.ExecutionContext
import com.yakindu.sct.model.sruntime.ExecutionEvent
import com.yakindu.sct.model.sruntime.ExecutionSlot
import com.yakindu.sct.model.sruntime.ExecutionVariable
import com.yakindu.sct.model.sruntime.SRuntimeFactory
import com.yakindu.sct.model.sruntime.impl.ExecutionContextImpl
import com.yakindu.sct.model.sruntime.impl.ExecutionVariableImpl
import org.apache.commons.scxml2.EventBuilder
import org.apache.commons.scxml2.SCXMLExecutor
import org.apache.commons.scxml2.SCXMLListener
import org.apache.commons.scxml2.TriggerEvent
import org.apache.commons.scxml2.model.EnterableState
import org.apache.commons.scxml2.model.Transition
import org.apache.commons.scxml2.model.TransitionTarget
import org.eclipse.emf.common.notify.Notification
import org.eclipse.emf.common.notify.impl.AdapterImpl
import org.eclipse.xtext.naming.IQualifiedNameProvider

import static com.yakindu.base.types.typesystem.ITypeSystem.*
import static com.yakindu.sct.model.sruntime.SRuntimePackage.Literals.*

/** 
 * 
 * @author andreas muelder - Initial contribution and API
 * 
 */
class BiDiExecutionContextAdapter extends ExecutionContextImpl implements SCXMLListener {

	@Inject IModelSequencer sequencer
	@Inject IExecutionContextInitializer initializer
	@Inject extension IExpressionInterpreter
	@Inject extension IQualifiedNameProvider
	extension SRuntimeFactory = SRuntimeFactory.eINSTANCE

	var protected SCXMLExecutor executor
	var Statechart statechart

	def init(Statechart statechart, SCXMLExecutor executor) {
		this.executor = executor
		this.statechart = statechart
		// TODO: Why is a ExecutionFlow needed by the IExecutionContextInitializer ?
		// A Statechart should be enough
		val executionFlow = sequencer.transform(statechart)
		initializer.initialize(this, executionFlow)

		executionFlow.initSequence.steps.filter(Execution).forEach[statement.evaluate(this)]
		executionFlow.staticInitSequence.steps.filter(Execution).forEach[statement.evaluate(this)]

		installAdapter()
		// Slot for SCXMLs built-in _event.data returns the last active event value
		slots += createCompositeSlot => [ cs |
			cs.name = "_event"
			cs.fqName = "_event"
			val slot = new ExecutionVariableImpl() {
				override getValue() {
					if (raisedEvents.isEmpty)
						return null
					raisedEvents.head.value
				}
			}
			slot.name = "data"
			slot.setFqName("_event.data")
			cs.slots += slot
		]
		// Slot for Math.floor
		slots += createCompositeSlot => [ cs |
			cs.name = "Math"
			cs.fqName = "Math"
			val slot = createExecutionVariable => [
				name = "floor"
				setFqName("Math.floor")
			]
			cs.slots += slot
		]
	}

	def installAdapter() {
		allSlots.forEach[adapt]

	}

	def dispatch adapt(ExecutionVariable it) {
		eAdapters() += new AdapterImpl {
			override notifyChanged(Notification msg) {
				if (msg.feature == EXECUTION_SLOT__VALUE) {
					executor.globalContext.set(it.name, msg.newValue)
				}
			}
		}
	}

	def dispatch adapt(ExecutionEvent it) {
		eAdapters() += new AdapterImpl {
			override notifyChanged(Notification msg) {
				if (msg.feature == EXECUTION_EVENT__RAISED) {
					if (msg.newBooleanValue) {
						val triggerEvent = switch (it.value) {
							case null: new EventBuilder(it.fqName, TriggerEvent.CALL_EVENT).build
							default: new EventBuilder(it.fqName, TriggerEvent.CALL_EVENT).data(it.value).build
						}
						raiseEvent(triggerEvent, it)
					}
				}
			}
		}
	}

	def protected raiseEvent(TriggerEvent it, ExecutionEvent ev) {
		new Thread() {
			override run() {
				executor.triggerEvent(it)
				ev.raised = false
			}
		}.start
	}

	def dispatch adapt(Object it) {
	}

	override synchronized onEntry(EnterableState it) {
		val state = it.id.findState
		if (state !== null) {
			getActiveStates() += state
		}
	}

	override synchronized onExit(EnterableState it) {
		val state = it.id.findState
		if (state !== null) {
			getActiveStates() -= state
		}
	}

	override synchronized onTransition(TransitionTarget source, TransitionTarget target, Transition transition,
		String string) {
		val t = findTransition(source, target, string)
		if (t !== null) {
			getExecutedElements.clear
			getExecutedElements += t
		}
	}

	protected def findTransition(TransitionTarget source, TransitionTarget target, String trigger) {
		if (trigger === null)
			return null
		try {
			val sourceTargetCandidates = findState(source.id)?.outgoingTransitions?.filter [
				it.target == findState(target.id)
			]
			if (sourceTargetCandidates === null)
				return null;
			return if (sourceTargetCandidates.size == 1) {
				sourceTargetCandidates.head
			} else {
				sourceTargetCandidates.findFirst[specification !== null && specification.contains(trigger)]
			}
		} catch (Exception ex) {
			ex.printStackTrace
		}
	}

	def Vertex findState(String name) {
		statechart.eAllContents.filter(Vertex).findFirst[it.fullyQualifiedName.toString == name]
	}
	
	def getAllVertices() {
		statechart.eAllContents.filter(Vertex).map[it.fullyQualifiedName.toString]
		
	}

	def restore(ExecutionContext that) {
		for (ExecutionSlot executionSlot : that.allSlots) {
			getSlot(executionSlot.getName())?.setValue(executionSlot.getValue());
		}
		val set = that.activeStates?.map[activeState|(activeState as RegularState).name].toSet
		set.forEach[activeStates += findState]
		executor.configuration = set
	}

	def protected dispatch type(Object it) { ANY }

	def protected dispatch type(Long it) { INTEGER }

	def protected dispatch type(Double it) { REAL }

	def protected dispatch type(Boolean it) { BOOLEAN }

	def protected dispatch type(String it) { STRING }

	def protected dispatch type(Void it) { VOID }

}
