/**
 * Copyright (c) 2022 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 * Contributors:
 * 	andreas muelder - itemis AG
 * 
 */
package com.yakindu.sct.types.lib

import com.google.inject.Inject
import com.google.inject.Singleton
import com.yakindu.base.expressions.parser.antlr.ExpressionsParser
import com.yakindu.base.types.ComplexType
import com.yakindu.base.types.Package
import com.yakindu.base.types.TypesFactory
import com.yakindu.base.types.typesystem.ITypeSystem
import java.io.StringReader
import org.eclipse.emf.common.notify.Adapter
import org.eclipse.emf.common.util.URI
import org.eclipse.emf.ecore.EObject
import org.eclipse.emf.ecore.resource.impl.ResourceImpl

/**
 * 
 * @author andreas muelder - Initial contribution and API
 * 
 */
@Singleton
class StatechartLibrary {

	public static final String STATEMACHINE_TYPES = "statemachine.types";

	extension TypesFactory factory = TypesFactory.eINSTANCE
	@Inject extension ITypeSystem ts
	var Package typesPackage;
	@Inject(optional=true) ExpressionsParser parser;

	public static val EVENT_DRIVEN_NAME = "EventDrivenStatemachine"
	public static val CYCLE_BASED_NAME = "CycleBasedStatemachine"
	public static val STATEMACHINE_NAME = "Statemachine"

	val resource = new ResourceImpl {
		override protected doUnload() {
			// Do no unload in memory resource
		}
	}

	def protected create() {
		typesPackage = createBaseTypes
		resource.URI = URI.createURI(STATEMACHINE_TYPES)
		resource.contents += typesPackage
	}

	def getTypesPackage() {
		if (typesPackage === null) {
			create();
		}
		return typesPackage
	}

	def getEventDriven() {
		getTypesPackage().member.findFirst[name == EVENT_DRIVEN_NAME] as ComplexType
	}

	def getCycleBased() {
		getTypesPackage().member.findFirst[name == CYCLE_BASED_NAME] as ComplexType
	}

	def getStatemachine() {
		getTypesPackage().member.findFirst[name == STATEMACHINE_NAME] as ComplexType
	}

	protected def createBaseTypes() {
		createPackage => [ package |
			package.name = ""

			var smType = createStatemachineType()

			package.member += smType

			package.member += createCycleBased(smType)
			package.member += createEventDriven(smType)
		]
	}

	protected def createStatemachineType() {
		val baseType = createComplexType => [
			name = STATEMACHINE_NAME

			features += createOperation => [
				name = "enter"
				doc(
					'''
						/**
						 * Enters the state machine. Sets the state machine into a defined state.
						*/
					'''
				)
				typeSpecifier = createTypeSpecifier => [
					type = ts.getType(ITypeSystem.VOID)
				]
			]
			features += createOperation => [
				name = "exit"
				doc(
					'''
						/**
						 * Exits the state machine. Leaves the state machine with a defined state.
						*/
					'''
				)
				typeSpecifier = createTypeSpecifier => [
					type = ts.getType(ITypeSystem.VOID)
				]
			]
			features += createOperation => [
				name = "isActive"
				doc(
					'''
						/**
						 * Checks whether the state machine is active. <br />
						 * A state machine is active if it has been entered. It is inactive if it has not been entered at all or if it has been exited.
						*/
					'''
				)
				typeSpecifier = createTypeSpecifier => [
					type = ts.getType(ITypeSystem.BOOLEAN)
				]
			]
			features += createOperation => [
				name = "isFinal"
				doc(
					'''
						/**
						 * Checks whether all active states are final. <br />
						 * If there are no active states then the state machine is considered being inactive. In this case this method returns <code>false</code>.
						 */
					'''
				)
				typeSpecifier = createTypeSpecifier => [
					type = ts.getType(ITypeSystem.BOOLEAN)
				]
			]
		]
		baseType.doc(
			'''
				/**
				 * Basic interface for state machines.
				*/
			'''
		)
		return baseType
	}

	protected def createCycleBased(ComplexType superType) {
		val cycleBased = createComplexType => [
			name = CYCLE_BASED_NAME

			features += createOperation => [
				name = "runCycle"
				doc(
					'''
						/**
						 * Start a run-to-completion cycle.
						*/
					'''
				)
				typeSpecifier = createTypeSpecifier => [
					type = ts.getType(ITypeSystem.VOID)
				]
			]
			superTypes += createTypeSpecifier => [
				type = superType
			]
		]
		cycleBased.doc(
			'''
				/**
				 * Interface for cycle-based state machines.
				 */
			'''
		)
		cycleBased
	}

	protected def createEventDriven(ComplexType superType) {
		val eventDriven = createComplexType => [
			name = EVENT_DRIVEN_NAME

			features += createOperation => [
				name = "triggerWithoutEvent"
				doc(
					'''
						/**
						 * Performs a state machine run-to-completion step without raising any event.
						 */
					'''
				)
				typeSpecifier = createTypeSpecifier => [
					type = ts.getType(ITypeSystem.VOID)
				]
			]
			superTypes += createTypeSpecifier => [
				type = superType
			]
		]

		eventDriven.doc(
			'''
				/**
				 * Interface for event-driven state machines.
				 */
			'''
		)

		return eventDriven
	}

	protected def doc(EObject it, String documentation) {
		if (parser !== null) {
			val doc = parser.parse(new StringReader(documentation));
			eAdapters += doc.rootNode as Adapter
		}
	}

}
