/**
 * Copyright (c) 2025 itemis AG - All rights Reserved
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * 
 */
package com.yakindu.sct.model.stext.concepts

import com.google.inject.Inject
import com.google.inject.Injector
import com.google.inject.Singleton
import com.yakindu.base.expressions.interpreter.IExpressionInterpreter
import com.yakindu.base.types.AnnotatableElement
import com.yakindu.base.types.AnnotatedElement
import com.yakindu.base.types.AnnotationType
import com.yakindu.base.types.TypesFactory
import com.yakindu.base.types.libraries.AbstractTypeLibrary
import com.yakindu.base.types.libraries.ITypeLibraryProvider
import com.yakindu.sct.model.sgraph.SGraphPackage
import com.yakindu.sct.model.sgraph.Statechart
import com.yakindu.sct.model.sruntime.SRuntimeFactory
import com.yakindu.sct.model.stext.stext.StextPackage
import org.eclipse.emf.common.util.URI
import org.eclipse.emf.ecore.EObject

/**
 * This class defines everything related to stext statechart annotations.
 * It provides extension methods which can be used to query statechart annotations defined by this class.
 */
@Singleton
class StatechartAnnotations {
	
	public static val LIBRARY = "create.statecharts"
	public static val LIBRARY_URI = URI.createURI(LIBRARY)

	public static val STATECHART_PACKAGE_NAME = "statecharts"

	public static final String CYCLE_BASED_ANNOTATION = "CycleBased";
	public static final String EVENT_DRIVEN_ANNOTATION = "EventDriven";
	public static final String PARENT_FIRST_ANNOTATION = "ParentFirstExecution";
	public static final String CHILD_FIRST_ANNOTATION = "ChildFirstExecution";
	public static final String SUPERSTEP_ANNOTATION = "SuperSteps";
	public static final String EVENTBUFFERING_ANNOTATION = "EventBuffering";
	public static final String DO_NOT_VALIDATE_INITIALIZATION_EXPRESSION_ANNOTATION = "DoNotValidateInitExpr";
	public static final String HALTABLE_ANNOTATION = "Haltable";
	public static final String EXPERIMENTAL_TYPES_ANNOTATION = "ExperimentalTypes";
	
	@Inject(optional = true)
	var IExpressionInterpreter interpreter;

	def boolean isCycleBased(Statechart statechart) {
		statechart.defines(CYCLE_BASED_ANNOTATION)
	}
	
	def long getCyclePeriod(EObject obj) {
		var cyclePeriod = 200l;
		if(obj instanceof AnnotatableElement) {
			val annotation = obj.getAnnotationOfType(CYCLE_BASED_ANNOTATION);
			if (annotation !== null && interpreter !== null) {
				cyclePeriod = interpreter.evaluate(annotation.getExpressions().get(0),
						SRuntimeFactory.eINSTANCE.createExecutionContext()) as Long
			}
		} else if(obj instanceof Statechart) {
			val annotation = obj.getAnnotationOfType(CYCLE_BASED_ANNOTATION);
			if (annotation !== null && interpreter !== null) {
				cyclePeriod = interpreter.evaluate(annotation.getExpressions().get(0),
						SRuntimeFactory.eINSTANCE.createExecutionContext()) as Long;
			}
		}
		return cyclePeriod;
	}
	
	def boolean isEventDriven(Statechart statechart) {
		! statechart.isCycleBased
	}
	
	def boolean isParentFirstExecution(Statechart statechart) {
		statechart.defines(PARENT_FIRST_ANNOTATION)
	}
	
	def boolean isChildFirstExecution(Statechart statechart) {
		! statechart.isParentFirstExecution
	}
	
	def boolean validateInitializationExpressions(Statechart statechart) {
		! statechart.defines(DO_NOT_VALIDATE_INITIALIZATION_EXPRESSION_ANNOTATION)
	}
	
	def boolean isInternalEventBuffer(Statechart statechart) {
		statechart.boolAnnotationParameter(EVENTBUFFERING_ANNOTATION, 1, true)
	}
	
	def boolean isInEventBuffer(Statechart statechart) {
		statechart.boolAnnotationParameter(EVENTBUFFERING_ANNOTATION, 0, true)
	}
	
	def boolean isSuperStep(Statechart statechart) {
		statechart.boolAnnotationParameter(SUPERSTEP_ANNOTATION, 0, false)
		
	}
	
	def boolean isExperimentalTypes(Statechart statechart) {
		statechart.boolAnnotationParameter(EXPERIMENTAL_TYPES_ANNOTATION, 0, false)
	}
	
	def boolean isHaltable(Statechart statechart) {
		statechart.defines(HALTABLE_ANNOTATION)
	}

	/**
	 * TODO refactor this to use annotation type from library.
	 */
	def void declareHaltable(Statechart statechart){
		if (statechart !== null && ! isHaltable(statechart)) {
			var annotation = TypesFactory.eINSTANCE.createAnnotation()
			var haltableAnnotationType = TypesFactory.eINSTANCE.createAnnotationType()
			haltableAnnotationType.setName(HALTABLE_ANNOTATION)
			annotation.setType(haltableAnnotationType)
			statechart.getAnnotations().add(annotation)
		}
	}
	
	def boolean boolAnnotationParameter(AnnotatedElement annotated, String annotation, int param, boolean defaultValue) {
		if (annotated === null)
			return false
		val eventBuffering = annotated.getAnnotationOfType(annotation)
		if(eventBuffering === null || eventBuffering.getExpressions().size() <= param || interpreter === null)
			return defaultValue
		return interpreter.evaluate(eventBuffering.getExpressions().get(param), SRuntimeFactory.eINSTANCE.createExecutionContext()) as Boolean
	}
	
	def protected defines(Statechart statechart, String annotationName) {
		statechart?.getAnnotationOfType(annotationName) !== null
	}
	/**
	 * Type library implementation which actually contributes the annotation types.
	 */
	@Singleton
	static class Library extends AbstractTypeLibrary {
		protected extension SGraphPackage sgraphPackage = SGraphPackage.eINSTANCE
		protected extension StextPackage stextPackage = StextPackage.eINSTANCE
		
		override protected createLibraryPackage() {
			STATECHART_PACKAGE_NAME._package  => [ package |
				package.member.addAll( #[
					EVENT_DRIVEN_ANNOTATION._annotationType => [
						statechartTargets
					],
					CYCLE_BASED_ANNOTATION._annotationType => [
						_param("period", _integer)
						statechartTargets
					],
					PARENT_FIRST_ANNOTATION._annotationType => [
						statechartTargets
					],
					CHILD_FIRST_ANNOTATION._annotationType => [
						statechartTargets
					],
					SUPERSTEP_ANNOTATION._annotationType => [
						_param("enabled", _boolean)
						statechartTargets
					],
					EVENTBUFFERING_ANNOTATION._annotationType => [
						_param("inEvents", _boolean)
						_param("localEvents", _boolean)
						statechartTargets
					],
					/* Haltable annotation is not publicly available yet ...  */
					// HALTABLE_ANNOTATION._annotationType => [
					//	statechartTargets
					//],
					EXPERIMENTAL_TYPES_ANNOTATION._annotationType => [
						_param("enabled", _boolean)
						statechartTargets
					]
				])
			]
		}
	
	
		def protected statechartTargets(AnnotationType it) {
			targets += #[ statechart, statechartSpecification ]
		}
		
		override getLibraryURI() {
			return LIBRARY_URI
		}
		
		
		static class Provider implements ITypeLibraryProvider {
			@Inject Injector injector
			override provide() {
				#[ injector.getInstance(StatechartAnnotations.Library) ]
			}
		}
		
	}
	

}